package cmd

import (
	"path/filepath"
	"syscall"
	"time"

	"a.yandex-team.ru/infra/rsm/perfmanager/internal/app"
	"a.yandex-team.ru/library/go/core/buildinfo"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/log/zap/logrotate"

	"github.com/spf13/cobra"
)

const (
	defaultLogName         = "perfmanager.log"
	defaultLogDir          = "/var/log/yandex-perf-manager"
	defaultPerfPath        = "/opt/yandex-perf-manager/perf"
	defaultLogLevel        = log.InfoLevel
	defaultIvalStatCollect = 60
	defaulIvalTopCollect   = 60
	defaultIvalTopPush     = 300
	logrotateSignal        = syscall.SIGHUP
)

var (
	debug            bool
	ivalStatCollect  int
	ivalTopCollect   int
	ivalTopPush      int
	logDir           string
	perfPath         string
	perfCgrpNamePrfx string
	perfCgrpNames    string
	logLevel         = defaultLogLevel
	logger           *zap.Logger
)

func Execute() {
	rootCmd := &cobra.Command{
		Use:   "perfmanager",
		Short: "Perf manager",
		Long: `Perf manager,
      Docs: https://a.yandex-team.ru/arc/trunk/arcadia/infra/rsm/perfmanager/README.md`,
		Version: buildinfo.Info.ProgramVersion,
		PersistentPreRunE: func(cmd *cobra.Command, args []string) error {
			return setupLogger()
		},
		Run: func(cmd *cobra.Command, args []string) {
			app.New(logger, time.Duration(ivalStatCollect)*time.Second, time.Duration(ivalTopCollect)*time.Second, time.Duration(ivalTopPush)*time.Second, perfPath).Run()
		},
	}

	if buildinfo.Info.ProgramVersion == "" {
		rootCmd.Version = "N/A"
	} else {
		rootCmd.Version = buildinfo.Info.ProgramVersion
	}

	rootCmd.PersistentFlags().BoolVarP(&debug, "debug", "d", false, "enable debug level")
	rootCmd.PersistentFlags().IntVar(&ivalStatCollect, "stat_collect_ival", defaultIvalStatCollect, "interval sampling")
	rootCmd.PersistentFlags().IntVar(&ivalTopCollect, "top_collect_ival", defaulIvalTopCollect, "interval sampling")
	rootCmd.PersistentFlags().IntVar(&ivalTopPush, "top_push_ival", defaultIvalTopPush, "interval pushing")
	rootCmd.PersistentFlags().StringVarP(&logDir, "log_dir", "l", defaultLogDir, "path to log files")
	rootCmd.PersistentFlags().StringVarP(&perfPath, "perf_path", "p", defaultPerfPath, "path to perf binary")

	if err := rootCmd.Execute(); err != nil {
		panic(err)
	}
}

func setupLogger() error {
	logPath := filepath.Join(logDir, defaultLogName)
	err := logrotate.RegisterLogrotateSink(logrotateSignal)
	if err != nil {
		return err
	}
	if debug {
		logLevel = log.DebugLevel
	}
	cfg := zap.ConsoleConfig(logLevel)
	// it'l rotate by signal
	cfg.OutputPaths = []string{"logrotate://" + logPath}
	cfg.ErrorOutputPaths = []string{logPath}
	cfg.DisableStacktrace = true
	logger, err = zap.New(cfg)
	if err != nil {
		return err
	}
	return nil
}
