package app

import (
	"context"
	"os"
	"os/signal"
	"sync"
	"syscall"
	"time"

	//"net/http"
	//_ "net/http/pprof"

	"a.yandex-team.ru/infra/rsm/perfmanager/internal/lb"
	"a.yandex-team.ru/infra/rsm/perfmanager/internal/perf"
	"a.yandex-team.ru/infra/rsm/perfmanager/internal/podsinfo"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue/log/corelogadapter"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/security/gideon/nuvault/pkg/nuvault"

	"github.com/ydb-platform/ydb-go-sdk/v3/credentials"
)

const (
	LBEndPoint = "logbroker.yandex.net"
	LBDB       = "/Root"
	CertPath   = "/etc/certs/capi.pem"
	YAVSecID   = "sec-01f4c4hkw16nzp780w87t6pwv7"
	YAVKey     = "robot-rtc-builder.lb_token"
	PSLBTopic  = "/rtcsysdev/perfmanager/stat"
	PTLBTopic  = "/rtcsysdev/perfmanager/top"
	//PSLBTopic = "/logbroker-playground/perfmanager/stat"
	//PTLBTopic = "/logbroker-playground/perfmanager/top"
)

var (
	stopSignals = []os.Signal{syscall.SIGTERM, syscall.SIGINT}
)

type app struct {
	l    *zap.Logger
	psLB *lb.Producer
	ps   *perf.Stat
	ptLB *lb.Producer
	pt   *perf.Top
	pi   *podsinfo.PodsInfo
}

func New(l *zap.Logger, si, tci, tpi time.Duration, perfPath string) *app {
	fqdn, err := os.Hostname()
	if err != nil {
		l.Fatalf("%s", err)
	}

	nuc, err := nuvault.NewClient(nuvault.WithMutualAuth(CertPath))
	if err != nil {
		l.Fatalf("%s", err)
	}
	defer func() { _ = nuc.Close() }()

	yav, err := nuc.GetSecret(context.TODO(), YAVSecID)
	if err != nil {
		l.Fatalf("%s", err)
	}
	token := credentials.NewAccessTokenCredentials(yav.Values[YAVKey])

	pi, err := podsinfo.New(l)
	if err != nil {
		l.Fatalf("%s", err)
	}
	pLock := &sync.Mutex{}
	ps, err := perf.NewStat(l, pLock, si, fqdn, perfPath, pi)
	if err != nil {
		l.Fatalf("%s", err)
	}
	pt, err := perf.NewTop(l, pLock, tci, tpi, fqdn, perfPath, pi)
	if err != nil {
		l.Fatalf("%s", err)
	}

	return &app{
		l:  l,
		ps: ps,
		psLB: lb.NewProducer(l, ps.Q, &persqueue.WriterOptions{
			Endpoint:       LBEndPoint,
			Credentials:    token,
			Database:       LBDB,
			Logger:         corelogadapter.New(l),
			Topic:          PSLBTopic,
			SourceID:       []byte(fqdn),
			Codec:          persqueue.Zstd,
			RetryOnFailure: true,
		}),
		pt: pt,
		ptLB: lb.NewProducer(l, pt.Q, &persqueue.WriterOptions{
			Endpoint:       LBEndPoint,
			Credentials:    token,
			Database:       LBDB,
			Logger:         corelogadapter.New(l),
			Topic:          PTLBTopic,
			SourceID:       []byte(fqdn),
			Codec:          persqueue.Zstd,
			RetryOnFailure: true,
		}),
		pi: pi,
	}
}

func (a *app) Run() {
	var wg sync.WaitGroup
	ctx, cancel := WithCancelOnSignal(context.Background(), a.l, stopSignals)
	//go func() {
	//  a.l.Errorf("%s", http.ListenAndServe("localhost:8080", nil))
	//}()

	wg.Add(1)
	go func() {
		defer wg.Done()
		a.l.Errorf("PodsInfo exiting by %s", a.pi.RunUpdateCache(ctx))
		cancel()
	}()

	wg.Add(1)
	go func() {
		defer wg.Done()
		a.l.Errorf("LBStat exiting by %s", a.psLB.Run(ctx))
		cancel()
	}()

	wg.Add(1)
	go func() {
		defer wg.Done()
		a.l.Errorf("PerfStat exiting by %s", a.ps.Run(ctx))
		cancel()
	}()

	wg.Add(1)
	go func() {
		defer wg.Done()
		a.l.Errorf("LBTop exiting by %s", a.ptLB.Run(ctx))
		cancel()
	}()

	wg.Add(1)
	go func() {
		defer wg.Done()
		a.l.Errorf("PerfTop exiting by %s", a.pt.Run(ctx))
		cancel()
	}()

	<-ctx.Done()
	wg.Wait()
}

func WaitForSignal(sigs []os.Signal) os.Signal {
	sig := make(chan os.Signal, 1)
	signal.Notify(sig, sigs...)
	return <-sig
}

func WithCancelOnSignal(ctx context.Context, l *zap.Logger, sigs []os.Signal) (context.Context, context.CancelFunc) {
	ctx, cancel := context.WithCancel(ctx)
	go func() {
		s := WaitForSignal(sigs)
		l.Errorf("Got signal: %s", s)
		cancel()
	}()
	return ctx, cancel
}
