package cpuid

import (
	"fmt"
	"io/ioutil"
	"regexp"
	"strconv"
	"strings"
)

const (
	cpuInfoPath = "/sys/devices/system/cpu/modalias"
)

var (
	// x86,ven0000fam0006mod0055
	cpuInfoRgxp = regexp.MustCompile("(.*),ven([0-9A-Fa-f]{4})fam([0-9A-Fa-f]{4})mod([0-9A-Fa-f]{4})")
)

type CPUID struct {
	Arch   string
	Vendor string
	Family uint64
	Model  uint64
	MArch  string
}

func Parse() (*CPUID, error) {
	buf, err := ioutil.ReadFile(cpuInfoPath)
	if err != nil {
		return nil, err
	}
	return parse(buf)
}

func parse(buf []byte) (*CPUID, error) {
	var err error
	data := strings.Split(strings.TrimSpace(string(buf)), ":")
	if len(data) < 3 {
		return nil, fmt.Errorf("failed to parse %s", cpuInfoPath)
	}
	cpuinfo := cpuInfoRgxp.FindStringSubmatch(data[2])
	if cpuinfo == nil {
		return nil, fmt.Errorf("failed to parse %s %s", cpuInfoPath, data[2])
	}
	cpuid := &CPUID{
		Arch: cpuinfo[1],
	}
	switch cpuinfo[2] {
	case "0000":
		cpuid.Vendor = "intel"
	case "0002":
		cpuid.Vendor = "amd"
	}
	cpuid.Family, err = strconv.ParseUint(cpuinfo[3], 16, 64)
	if err != nil {
		return nil, err
	}
	cpuid.Model, err = strconv.ParseUint(cpuinfo[4], 16, 64)
	if err != nil {
		return nil, err
	}
	if cpuid.Vendor == "intel" {
		switch {
		case cpuid.Model == 44 && cpuid.Family == 6:
			cpuid.MArch = "westmere"
		case cpuid.Model == 45 && cpuid.Family == 6:
			cpuid.MArch = "sandybridge"
		case cpuid.Model == 62 && cpuid.Family == 6:
			cpuid.MArch = "ivybridge"
		case (cpuid.Model == 79 || cpuid.Model == 86) && cpuid.Family == 6:
			cpuid.MArch = "broadwell"
		case cpuid.Model == 85 && cpuid.Family == 6:
			cpuid.MArch = "skylake"
		case cpuid.Model == 106 && cpuid.Family == 6:
			cpuid.MArch = "icelake"
		}
		// https://en.wikichip.org/wiki/amd/cpuid#Family_23_.2817h.29
	} else if cpuid.Vendor == "amd" {
		switch {
		case cpuid.Family == 23 && cpuid.Model == 1:
			cpuid.MArch = "zen"
		case cpuid.Family == 23 && cpuid.Model == 49:
			cpuid.MArch = "zen2"
		}
	}
	return cpuid, nil
}
