package lb

import (
	"context"
	"encoding/json"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/library/go/core/log/zap"
)

type Producer struct {
	Q    chan json.Marshaler
	l    *zap.Logger
	w    persqueue.Writer
	opts *persqueue.WriterOptions
}

func NewProducer(l *zap.Logger, q chan json.Marshaler, opts *persqueue.WriterOptions) *Producer {
	return &Producer{
		Q:    q,
		l:    l,
		opts: opts,
	}
}

func (p *Producer) Run(ctx context.Context) error {
	var err error
	p.w = persqueue.NewWriter(*p.opts)
	init, err := p.w.Init(ctx)
	if err != nil {
		return err
	}
	defer p.w.Close()
	seqNo := init.MaxSeqNo + 1

	go func(w persqueue.Writer) {
		for rsp := range w.C() {
			switch m := rsp.(type) {
			case *persqueue.Ack:
				p.l.Debugf("Ack seqNo: %v", m.SeqNo)
			case *persqueue.Issue:
				p.l.Warnf("Write error %s", m.Err)
			}
		}
	}(p.w)

	for {
		select {
		case m := <-p.Q:
			buf, err := json.Marshal(m)
			if err != nil {
				p.l.Errorf("%s", err)
				return err
			}
			p.l.Debugf("%s", buf)
			msg := &persqueue.WriteMessage{Data: buf}
			msg.WithSeqNo(seqNo)
			if err := p.w.Write(msg); err != nil {
				p.l.Errorf("Writer terminated: %s", err)
			}
			seqNo++
		case <-ctx.Done():
			return ctx.Err()
		}
	}
}
