package perf

import (
	"fmt"
	"strconv"
)

const (
	PipelineWidth = 4
)

var (
	CommonEvents = []string{
		"{major-faults,minor-faults}",
		"{migrations,cs,sched:sched_wakeup}",
		"{instructions,cycles,task-clock}",
		"{L1-dcache-load-misses,L1-icache-load-misses,dTLB-load-misses,iTLB-load-misses}",
	}
)

type MArchMetrics struct {
	// TopDownL1
	// How good we are at fetching, decoding instructions and feeding them to the rest of the pipeline
	// https://easyperf.net/blog/2018/12/29/Understanding-IDQ_UOPS_NOT_DELIVERED
	CalcFrontendBound  func(ed eventsData) float64
	CalcBadSpeculation func(ed eventsData) float64
	CalcRetiring       func(ed eventsData) float64
	CalcBackendBound   func(ed eventsData) float64
	CalcLLi            func(ed eventsData) float64
	CalcLLC            func(ed eventsData) float64
	Events             []string
}

var (
	MArchMetricsMap = map[string]*MArchMetrics{
		"ivybridge": &MArchMetrics{
			CalcFrontendBound:  calcIntelFrontendBound,
			CalcBadSpeculation: calcIntelBadSpeculation,
			CalcRetiring:       calcIntelRetiring,
			CalcBackendBound:   calcIntelBackendBound,
			CalcLLi:            calcIntelLLi,
			CalcLLC:            calcIntelLLC,
			Events: []string{
				"{LLC-load-misses,mem_uops_retired.lock_loads}",
				"{idq_uops_not_delivered.core,uops_issued.any,uops_retired.retire_slots}",
			},
		},
		"skylake": &MArchMetrics{
			CalcFrontendBound:  calcIntelFrontendBound,
			CalcBadSpeculation: calcIntelBadSpeculation,
			CalcRetiring:       calcIntelRetiring,
			CalcBackendBound:   calcIntelBackendBound,
			CalcLLi:            calcUndefined,
			CalcLLC:            calcIntelLLC,
			Events: []string{
				"{LLC-load-misses,idq_uops_not_delivered.core,uops_issued.any,uops_retired.retire_slots}",
			},
		},
		"westmere": &MArchMetrics{
			CalcFrontendBound:  calcUndefined,
			CalcBadSpeculation: calcIntelBadSpeculation,
			CalcRetiring:       calcIntelRetiring,
			CalcBackendBound:   calcUndefined,
			CalcLLi:            calcUndefined,
			CalcLLC:            calcIntelLLC,
			Events: []string{
				"{LLC-load-misses,uops_issued.any,uops_retired.retire_slots}",
			},
		},
		"broadwell": &MArchMetrics{
			CalcFrontendBound:  calcIntelFrontendBound,
			CalcBadSpeculation: calcIntelBadSpeculation,
			CalcRetiring:       calcIntelRetiring,
			CalcBackendBound:   calcIntelBackendBound,
			CalcLLi:            calcIntelLLi,
			CalcLLC:            calcIntelLLC,
			Events: []string{
				"{LLC-load-misses,mem_uops_retired.lock_loads}",
				"{idq_uops_not_delivered.core,uops_issued.any,uops_retired.retire_slots}",
			},
		},
		"sandybridge": &MArchMetrics{
			CalcFrontendBound:  calcIntelFrontendBound,
			CalcBadSpeculation: calcIntelBadSpeculation,
			CalcRetiring:       calcIntelRetiring,
			CalcBackendBound:   calcIntelBackendBound,
			CalcLLi:            calcIntelLLi,
			CalcLLC:            calcIntelLLC,
			Events: []string{
				"{LLC-load-misses,mem_uops_retired.lock_loads}",
				"{idq_uops_not_delivered.core,uops_issued.any,uops_retired.retire_slots}",
			},
		},
		"icelake": &MArchMetrics{
			CalcFrontendBound: calcIntelFrontendBound,
			CalcBadSpeculation: func(ed eventsData) float64 {
				return div((ed.Get("uops_issued.any") - ed.Get("uops_retired.slots")), PipelineWidth*ed.Get("cycles"))
			},
			CalcRetiring: func(ed eventsData) float64 {
				return div(ed.Get("uops_retired.slots"), PipelineWidth*ed.Get("cycles"))
			},
			CalcBackendBound: func(ed eventsData) float64 {
				var res float64
				notBe := div(ed.Get("idq_uops_not_delivered.core"), PipelineWidth*ed.Get("cycles")) + div((ed.Get("uops_issued.any")-ed.Get("uops_retired.slots")), PipelineWidth*ed.Get("cycles")) + div(ed.Get("uops_retired.slots"), PipelineWidth*ed.Get("cycles"))
				if notBe > 0 {
					res = 1 - notBe
				}
				return res
			},
			CalcLLi: calcUndefined,
			CalcLLC: calcIntelLLC,
			Events: []string{
				"{LLC-load-misses,idq_uops_not_delivered.core,uops_issued.any,uops_retired.slots}",
			},
		},
		"zen": &MArchMetrics{
			CalcFrontendBound:  calcAmdFrontendBound,
			CalcBadSpeculation: calcUndefined,
			CalcRetiring:       calcUndefined,
			CalcBackendBound:   calcAmdBackendBound,
			CalcLLi:            calcUndefined,
			CalcLLC:            calcUndefined,
			Events: []string{
				"{stalled-cycles-frontend,stalled-cycles-backend}",
			},
		},
		"zen2": &MArchMetrics{
			CalcFrontendBound:  calcAmdFrontendBound,
			CalcBadSpeculation: calcUndefined,
			CalcRetiring:       calcUndefined,
			CalcBackendBound:   calcAmdBackendBound,
			CalcLLi:            calcUndefined,
			CalcLLC:            calcUndefined,
			Events: []string{
				"{stalled-cycles-frontend,stalled-cycles-backend}",
			},
		},
	}
)

func init() {
	for _, v := range MArchMetricsMap {
		v.Events = append(v.Events, CommonEvents...)
	}
}

func div(x, y float64) (res float64) {
	if x > 0 && y > 0 {
		res = x / y
	}
	return
}

func pki(n float64) float64 {
	return round(n*1000, "%.2f")
}

func round(n float64, format string) (res float64) {
	res, _ = strconv.ParseFloat(fmt.Sprintf(format, n), 64)
	return
}

func perc(n float64) float64 {
	return round(100*n, "%.1f")
}

func calcUndefined(ed eventsData) float64 {
	return 0
}

func calcIPC(d eventsData) float64 {
	return div(d.Get("instructions"), d.Get("cycles"))
}

func calcCS(d eventsData) float64 {
	return div(d.Get("cs"), d.Get("task-clock"))
}

func calcMinPgFt(d eventsData) float64 {
	return div(d.Get("minor-faults"), d.Get("task-clock"))
}

func calcMajPgFt(d eventsData) float64 {
	return div(d.Get("major-faults"), d.Get("task-clock"))
}

func calcMigrations(d eventsData) float64 {
	return div(d.Get("migrations"), d.Get("task-clock"))
}

func calcWakeUp(d eventsData) float64 {
	return div(d.Get("sched:sched_wakeup"), d.Get("task-clock"))
}

func calcTLBi(ed eventsData) float64 {
	return div(ed.Get("iTLB-load-misses"), ed.Get("instructions"))
}

func calcTLBd(ed eventsData) float64 {
	return div(ed.Get("dTLB-load-misses"), ed.Get("instructions"))
}

func calcL1i(ed eventsData) float64 {
	return div(ed.Get("L1-icache-load-misses"), ed.Get("instructions"))
}

func calcL1d(ed eventsData) float64 {
	return div(ed.Get("L1-dcache-load-misses"), ed.Get("instructions"))
}

// Intel
func calcIntelLLC(ed eventsData) float64 {
	return div(ed.Get("LLC-load-misses"), ed.Get("instructions"))
}

func calcIntelLLi(ed eventsData) float64 {
	return div(ed.Get("mem_uops_retired.lock_loads"), ed.Get("instructions"))
}

func calcIntelFrontendBound(ed eventsData) float64 {
	return div(ed.Get("idq_uops_not_delivered.core"), PipelineWidth*ed.Get("cycles"))
}

func calcIntelBadSpeculation(ed eventsData) float64 {
	// TODO: should go to SMT metrics
	// int_misc.recovery_cycles_any doesn't work correctly with HT
	// see https://st.yandex-team.ru/KERNEL-617#60f98b6d6dd10a646ce25502
	// return div((d.Get("uops_issued.any") - d.Get("uops_retired.retire_slots") + PipelineWidth * d.Get("int_misc.recovery_cycles_any")), PipelineWidth * d.Get("CPU_CLK_UNHALTED"))
	return div((ed.Get("uops_issued.any") - ed.Get("uops_retired.retire_slots")), PipelineWidth*ed.Get("cycles"))
}

func calcIntelRetiring(ed eventsData) float64 {
	return div(ed.Get("uops_retired.retire_slots"), PipelineWidth*ed.Get("cycles"))
}

func calcIntelBackendBound(ed eventsData) (res float64) {
	notBe := div(ed.Get("idq_uops_not_delivered.core"), PipelineWidth*ed.Get("cycles")) + div((ed.Get("uops_issued.any")-ed.Get("uops_retired.retire_slots")), PipelineWidth*ed.Get("cycles")) + div(ed.Get("uops_retired.retire_slots"), PipelineWidth*ed.Get("cycles"))
	if notBe > 0 {
		res = 1 - notBe
	}
	return res
}

// AMD
func calcAmdFrontendBound(ed eventsData) float64 {
	return div(ed.Get("stalled-cycles-frontend"), ed.Get("cycles"))
}

func calcAmdBackendBound(ed eventsData) float64 {
	return div(ed.Get("stalled-cycles-backend"), ed.Get("cycles"))
}
