package perf

import (
	"bytes"
	"context"
	"encoding/csv"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"strconv"
	"strings"
	"sync"
	"time"

	"a.yandex-team.ru/infra/rsm/perfmanager/internal/cpuid"
	"a.yandex-team.ru/infra/rsm/perfmanager/internal/podsinfo"
	"a.yandex-team.ru/library/go/core/log/zap"
)

const (
	CsvComma      = '|'
	CsvMetricName = 2
	CsvCgrpName   = 3
	CsvAbsValue   = 0
	CsvPercValue  = 6
	QueueSize     = 1
	NMIPath       = "/proc/sys/kernel/nmi_watchdog"
)

var (
	defaultCollectTimeout = 5 * time.Second
)

type Stat struct {
	l            *zap.Logger
	ic           time.Duration
	fqdn         string
	perfPath     string
	perfArgs     []string
	marchMetrics *MArchMetrics
	pi           *podsinfo.PodsInfo
	Q            chan json.Marshaler
	pLock        *sync.Mutex
}

func NewStat(l *zap.Logger, pLock *sync.Mutex, ic time.Duration, fqdn, perfPath string, pi *podsinfo.PodsInfo) (*Stat, error) {
	cpuid, err := cpuid.Parse()
	if err != nil {
		return nil, err
	}
	l.Infof("CPU Microarchitecture: %s %s\n", cpuid.Vendor, cpuid.MArch)
	march, ok := MArchMetricsMap[cpuid.MArch]
	if !ok {
		return nil, fmt.Errorf("perf metrics not found for: %s %s", cpuid.Vendor, cpuid.MArch)
	}
	return &Stat{
		l:        l,
		ic:       ic,
		fqdn:     fqdn,
		perfPath: perfPath,
		perfArgs: []string{
			"stat", "-a", "-x", string(CsvComma),
			"-e", strings.Join(march.Events, ","),
			"--timeout", strconv.FormatInt(defaultCollectTimeout.Milliseconds(), 10),
			"--for-each-cgroup",
		},
		marchMetrics: march,
		pi:           pi,
		Q:            make(chan json.Marshaler, QueueSize),
		pLock:        pLock,
	}, nil
}

func (s *Stat) Run(ctx context.Context) error {
	ic := time.NewTicker(s.ic)
	defer ic.Stop()

	for {
		select {
		case <-ic.C:
			if err := s.push(ctx); err != nil {
				s.l.Errorf("%s\n", err)
			}
		case <-ctx.Done():
			close(s.Q)
			return ctx.Err()
		}
	}
}

func (s *Stat) push(ctx context.Context) error {
	cgNames := s.pi.CgNames()
	s.l.Debugf("Found containers: %s", cgNames)
	if len(cgNames) == 0 {
		return nil
	}

	ts := time.Now()
	// TODO: Temporarily disabling NMI logic
	//if err := setNMI("0"); err != nil {
	//	return err
	//}
	buf, err := s.runStat(ctx, cgNames)
	if err != nil {
		return err
	}
	//if err = setNMI("1"); err != nil {
	//	return err
	//}
	srs, err := statRecordsParse(ts, s.fqdn, s.marchMetrics, s.pi, buf)
	if err != nil {
		s.l.Errorf("%s", err)
		return nil
	}
	for _, i := range srs {
		s.Q <- i
	}
	return nil
}

func (s *Stat) runStat(ctx context.Context, cgNames []string) (*bytes.Buffer, error) {
	perfBinArgs := append(s.perfArgs, strings.Join(cgNames, ","))
	s.l.Infof("Running perf stat: %s %s", s.perfPath, strings.Join(perfBinArgs, " "))
	s.pLock.Lock()
	stdout, stderr, err := cmdRun(ctx, 3*defaultCollectTimeout, 4*defaultCollectTimeout, true, s.perfPath, perfBinArgs...)
	s.pLock.Unlock()
	if err != nil {
		s.l.Errorf("Stdout:\n%s", stdout)
		s.l.Errorf("Stderr:\n%s", stderr)
		return nil, err
	}
	s.l.Debugf("Stdout:\n%s", stdout)
	s.l.Debugf("Stderr:\n%s", stderr)
	return stderr, nil
}

type StatRecord struct {
	TS           string  `json:"ts"`
	FQDN         string  `json:"fqdn"`
	Name         string  `json:"cgrp"`
	CtName       string  `json:"ctname"`
	IPC          float64 `json:"ipc"`
	CS           float64 `json:"cs"`           // K/sec
	WakeUp       float64 `json:"wakeup"`       // K/sec
	MinPgFt      float64 `json:"minPgFt"`      // K/sec
	MajPgFt      float64 `json:"majPgFt"`      // K/sec
	Migrations   float64 `json:"migrations"`   // K/sec
	Fe           float64 `json:"fe"`           // %
	Be           float64 `json:"be"`           // %
	Ret          float64 `json:"ret"`          // %
	Bad          float64 `json:"bad"`          // %
	TLBi         float64 `json:"iTLB"`         // PKI
	TLBd         float64 `json:"dTLB"`         // PKI
	L1i          float64 `json:"L1i"`          // PKI
	L1d          float64 `json:"L1d"`          // PKI
	LLC          float64 `json:"LLC"`          // PKI
	LLi          float64 `json:"lli"`          // PKI
	NetRxBytes   uint64  `json:"netrxbytes"`   // PerSec
	NetTxBytes   uint64  `json:"nettxbytes"`   // PerSec
	NetRxPkts    uint64  `json:"netrxpkts"`    // PerSec
	NetTxPkts    uint64  `json:"nettxpkts"`    // PerSec
	IOReadBytes  uint64  `json:"ioreadbytes"`  // PerSec
	IOWriteBytes uint64  `json:"iowritebytes"` // PerSec
	CPUUsage     uint64  `json:"cpuusage"`     // Absoulte
}

func StatRecordParse(ts time.Time, fqdn string, march *MArchMetrics, pod *podsinfo.Pod, ed eventsData) *StatRecord {
	return &StatRecord{
		TS:           fmt.Sprintf("%d", ts.Unix()),
		FQDN:         fqdn,
		Name:         pod.Name,
		CtName:       pod.CtName,
		IPC:          round(calcIPC(ed), "%.2f"),
		CS:           round(calcCS(ed), "%.3f"),
		MinPgFt:      round(calcMinPgFt(ed), "%.3f"),
		MajPgFt:      round(calcMajPgFt(ed), "%.3f"),
		Migrations:   round(calcMigrations(ed), "%.3f"),
		WakeUp:       round(calcWakeUp(ed), "%.3f"),
		TLBi:         pki(calcTLBi(ed)),
		TLBd:         pki(calcTLBd(ed)),
		L1i:          pki(calcL1i(ed)),
		L1d:          pki(calcL1d(ed)),
		LLC:          pki(march.CalcLLC(ed)),
		LLi:          pki(march.CalcLLi(ed)),
		Fe:           perc(march.CalcFrontendBound(ed)),
		Be:           perc(march.CalcBackendBound(ed)),
		Bad:          perc(march.CalcBadSpeculation(ed)),
		Ret:          perc(march.CalcRetiring(ed)),
		NetRxBytes:   pod.GetNetRxBytes(),
		NetTxBytes:   pod.GetNetTxBytes(),
		NetRxPkts:    pod.GetNetRxPkts(),
		NetTxPkts:    pod.GetNetTxPkts(),
		IOReadBytes:  pod.GetIOReadBytes(),
		IOWriteBytes: pod.GetIOWriteBytes(),
		CPUUsage:     pod.GetCPUUsage(),
	}
}

func (sd *StatRecord) MarshalJSON() ([]byte, error) {
	return json.Marshal(map[string]interface{}{
		"ts":           sd.TS,
		"fqdn":         sd.FQDN,
		"cgrp":         sd.Name,
		"ctname":       sd.CtName,
		"ipc":          sd.IPC,
		"cs":           sd.CS,
		"wakeup":       sd.WakeUp,
		"minPgFt":      sd.MinPgFt,
		"majPgFt":      sd.MajPgFt,
		"migrations":   sd.Migrations,
		"fe":           sd.Fe,
		"be":           sd.Be,
		"ret":          sd.Ret,
		"bad":          sd.Bad,
		"iTLB":         sd.TLBi,
		"dTLB":         sd.TLBd,
		"L1i":          sd.L1i,
		"L1d":          sd.L1d,
		"LLC":          sd.LLC,
		"lli":          sd.LLi,
		"netrxbytes":   sd.NetRxBytes,
		"nettxbytes":   sd.NetTxBytes,
		"netrxpkts":    sd.NetRxPkts,
		"nettxpkts":    sd.NetTxPkts,
		"ioreadbytes":  sd.IOReadBytes,
		"iowritebytes": sd.IOWriteBytes,
		"cpuusage":     sd.CPUUsage,
	})
}

type StatRecords []*StatRecord

func statRecordsParse(ts time.Time, fqdn string, march *MArchMetrics, pi *podsinfo.PodsInfo, buf *bytes.Buffer) (StatRecords, error) {
	var (
		ed  eventsData
		edm = make(map[string]eventsData)
		srs = []*StatRecord{}
	)

	r := csv.NewReader(buf)
	r.Comma = CsvComma
	for {
		line, err := r.Read()
		if err == io.EOF {
			break
		}
		if err != nil {
			continue
		}
		if len(line) < 7 {
			continue
		}
		cgName := line[CsvCgrpName]
		if cgName == "" {
			continue
		}
		if _, ok := edm[cgName]; !ok {
			ed = make(eventsData)
			edm[cgName] = ed
		}
		metricName := line[CsvMetricName]
		val, err := strconv.ParseFloat(line[CsvAbsValue], 64)
		if err != nil {
			val = 0
		}
		ed.Set(metricName, val)
	}
	for cgName, data := range edm {
		pod := pi.FromCgName(cgName)
		if pod == nil {
			continue
		}
		srs = append(srs, StatRecordParse(ts, fqdn, march, pod, data))
	}
	return srs, nil
}

type eventsData map[string]float64

func (e *eventsData) Set(m string, v float64) {
	(*e)[m] = v
}

func (e *eventsData) Get(m string) float64 {
	return (*e)[m]
}

func parseCgrpName(prfx, name string) string {
	return strings.Replace(name, prfx, "", 1)
}

func setNMI(data string) error {
	return ioutil.WriteFile(NMIPath, []byte(data), 0)
}
