package perf

import (
	"bufio"
	"bytes"
	"context"
	"crypto/sha1"
	"encoding/json"
	"fmt"
	"sort"
	"strconv"
	"strings"
	"sync"
	"time"

	"a.yandex-team.ru/infra/rsm/perfmanager/internal/podsinfo"
	"a.yandex-team.ru/library/go/core/log/zap"
)

const (
	PerfDataPath                = "/tmp/yandex-perf-manager.data"
	MaxFuncNameLength int       = 128
	Freq                        = "99"
	FieldsDelim                 = "##"
	UserSpacePrfx               = "[.]"
	KernelSpacePrfx             = "[k]"
	KernelSpace       SpaceType = iota + 1
	UserSpace
	binSleepPath = "/bin/sleep"
)

var (
	FuncLimit  = 500
	ReportArgs = []string{
		"report", "--stdio",
		"--stdio-color", "never",
		"--no-skip-empty", "-i", PerfDataPath,
		"-F", "period,comm,dso,symbol",
		"--field-separator", FieldsDelim}
	// linux/tools/perf/util/symbol.c
	IdleFuncs = []string{
		"arch_cpu_idle", "cpu_idle", "cpu_startup_entry",
		"intel_idle", "default_idle", "native_safe_halt",
		"enter_idle", "exit_idle", "mwait_idle", "mwait_idle_with_hints",
		"poll_idle", "ppc64_runlatch_off", "pseries_dedicated_idle_sleep",
	}
)

type SpaceType int

type Top struct {
	l          *zap.Logger
	cIval      time.Duration // collect interval
	pIval      time.Duration // push interval
	fqdn       string
	binPath    string
	recordArgs []string
	pi         *podsinfo.PodsInfo
	trs        *TopRecords
	Q          chan json.Marshaler
	pLock      *sync.Mutex
}

func NewTop(l *zap.Logger, pLock *sync.Mutex, cIval, pIval time.Duration, fqdn string, binPath string, pi *podsinfo.PodsInfo) (*Top, error) {
	return &Top{
		l:       l,
		cIval:   cIval,
		pIval:   pIval,
		fqdn:    fqdn,
		binPath: binPath,
		recordArgs: []string{
			"record",
			"-F", Freq,
			"-o", PerfDataPath,
		},
		pi:    pi,
		trs:   NewTopRecords(),
		Q:     make(chan json.Marshaler, QueueSize),
		pLock: pLock,
	}, nil
}

func (p *Top) Run(ctx context.Context) error {
	cTick := time.NewTicker(p.cIval)
	pTick := time.NewTicker(p.pIval)
	defer cTick.Stop()
	defer pTick.Stop()

	for {
		select {
		case <-cTick.C:
			if err := p.collect(ctx); err != nil {
				p.l.Errorf("%s\n", err)
			}
		case <-pTick.C:
			p.push(ctx)
		case <-ctx.Done():
			close(p.Q)
			return ctx.Err()
		}
	}
}

func (p *Top) collect(ctx context.Context) error {
	cgNames := p.pi.CgNames()
	p.l.Debugf("Found containers: %s", cgNames)
	if len(cgNames) == 0 {
		return nil
	}

	ts := time.Now()
	if err := p.runRecord(ctx, cgNames); err != nil {
		return err
	}

	buf, err := p.runReport(ctx)
	if buf == nil {
		return err
	}
	// TODO: KERNEL-872
	//cgNames = append(cgNames, "kernel")
	rs, err := recordsRead(ts, p.fqdn, cgNames, p.pi, buf)
	if err != nil {
		return err
	}
	p.trs.update(rs)
	return nil
}

func (p *Top) push(ctx context.Context) {
	p.l.Infof("Running push")
	p.trs.mux.Lock()
	defer p.trs.mux.Unlock()
	for _, tr := range p.trs.trFromName {
		// skip empty cgrps (without top funcs)
		if len(tr.Data) > 0 {
			// push only first FuncLimit funcs
			tr.trim(FuncLimit)
			p.Q <- tr
		}
		delete(p.trs.trFromName, tr.Name)
	}
}

func (p *Top) runRecord(ctx context.Context, cgNames []string) error {
	recordArgs := p.recordArgs
	for _, cgName := range cgNames {
		recordArgs = append(recordArgs, "-e", "cycles", "-G", cgName)
	}
	// TODO: KERNEL-872
	//recordArgs = append(recordArgs, "-e", "cycles", "-a", binSleepPath, fmt.Sprintf("%.0f", defaultCollectTimeout.Seconds()))
	recordArgs = append(recordArgs, "-a", binSleepPath, fmt.Sprintf("%.0f", defaultCollectTimeout.Seconds()))
	p.l.Infof("Running perf record: %s %s", p.binPath, strings.Join(recordArgs, " "))
	p.pLock.Lock()
	stdout, stderr, err := cmdRun(ctx, 3*defaultCollectTimeout, 4*defaultCollectTimeout, true, p.binPath, recordArgs...)
	p.pLock.Unlock()
	if err != nil {
		p.l.Errorf("Stdout:\n%s", stdout)
		p.l.Errorf("Stderr:\n%s", stderr)
		return err
	}
	p.l.Debugf("Stdout:\n%s", stdout)
	p.l.Debugf("Stderr:\n%s", stderr)
	return nil
}

func (p *Top) runReport(ctx context.Context) (*bytes.Buffer, error) {
	p.l.Infof("Running perf report: %s %s", p.binPath, strings.Join(ReportArgs, " "))
	stdout, stderr, err := cmdRun(ctx, 6*defaultCollectTimeout, 7*defaultCollectTimeout, false, p.binPath, ReportArgs...)
	if err == nil && stderr.Len() > 0 {
		err = fmt.Errorf("%s", stderr)
	}
	if err != nil {
		//p.l.Errorf("Stdout:\n%s", stdout)
		p.l.Errorf("Stderr:\n%s", stderr)
		return nil, err
	}
	p.l.Debugf("Stdout:\n%s", stdout)
	p.l.Debugf("Stderr:\n%s", stderr)
	return stdout, nil
}

type TopData struct {
	EvCount uint64 `json:"evcount"`
	DSO     string `json:"dso"`
	Comm    string `json:"comm"`
	Sym     string `json:"func"`
	cs      string
	st      SpaceType
}

func NewTopData(evCount uint64, dso, comm, sym string, st SpaceType) *TopData {
	if st == KernelSpace {
		comm = ""
		dso = "kernel"
		if symIsIdle(sym) {
			sym = "idle"
		}
	}
	if strings.HasPrefix(dso, "[JIT") {
		dso = "JIT"
	} else if strings.HasSuffix(dso, ".tmp") || strings.HasSuffix(dso, ".vbt") {
		// libbrotli.so-165399725881915752116802613052755.tmp
		// 9VT2LOtc.vbt
		dso = "unknown"
	}
	if strings.HasPrefix(sym, "0x") {
		sym = "unknown"
	}
	// 0.59%  aed3a339-beaa2d23-79-f50195c9  [.] 0x0000000004ef6a27
	if sym == "unknown" && dso != "JIT" {
		dso = "unknown"
	}
	// trim function name
	sym = sym[:min(len(sym), MaxFuncNameLength)]
	return &TopData{
		EvCount: evCount,
		DSO:     dso,
		Comm:    comm,
		Sym:     sym,
		cs:      checksum(dso + comm + sym),
		st:      st,
	}
}

// evCount     comm              dso                 st     sym
// 134392356 ##Connection:1    ##[JIT] tid 970547  ##[.] 0x00007efd2ec440fa
func TopDataParse(s string, l []string) (*TopData, error) {
	var (
		st   SpaceType
		dso  string
		comm string
		sym  string
	)
	if len(s) == 0 || strings.HasPrefix(s, "#") {
		return nil, fmt.Errorf("skipping comment out line: %s", s)
	}
	evCount, err := strconv.ParseUint(strings.TrimSpace(l[0]), 10, 64)
	if err != nil {
		return nil, err
	}
	comm = strings.TrimSpace(l[1])
	dso = strings.TrimSpace(l[2])
	symTmp := strings.SplitN(strings.TrimSpace(l[3]), " ", 2)
	stTmp := symTmp[0]
	if stTmp == UserSpacePrfx {
		st = UserSpace
	} else if stTmp == KernelSpacePrfx {
		st = KernelSpace
	} else {
		return nil, fmt.Errorf("unsupported format: %s", s)
	}
	sym = strings.TrimSpace(symTmp[1])
	return NewTopData(evCount, dso, comm, sym, st), nil
}

type TopRecord struct {
	TS             time.Time  `json:"ts"`
	FQDN           string     `json:"fqdn"`
	Name           string     `json:"cgrp"`
	EvCount        uint64     `json:"evcount"`
	EvCountSkipped uint64     `json:"evcount_skipped"`
	Data           []*TopData `json:"data"`
	tdFromCs       map[string]*TopData
}

func NewTopRecord(ts time.Time, fqdn string, name string, evCount uint64) *TopRecord {
	return &TopRecord{
		TS:       ts,
		FQDN:     fqdn,
		Name:     name,
		EvCount:  evCount,
		Data:     []*TopData{},
		tdFromCs: map[string]*TopData{},
	}
}

func (tr *TopRecord) update(trNew *TopRecord) {
	tr.TS = trNew.TS
	tr.EvCount += trNew.EvCount
	tr.EvCountSkipped += trNew.EvCountSkipped
	for _, tdNew := range trNew.Data {
		tr.add(tdNew)
	}
}

func (tr *TopRecord) add(tdNew *TopData) {
	if td, ok := tr.tdFromCs[tdNew.cs]; ok {
		td.EvCount += tdNew.EvCount
	} else {
		tr.Data = append(tr.Data, tdNew)
		tr.tdFromCs[tdNew.cs] = tdNew
	}
}

func (tr *TopRecord) trim(l int) {
	funcLimit := minInt(len(tr.Data), l)
	sort.Slice(tr.Data, func(i, j int) bool {
		return tr.Data[i].EvCount > tr.Data[j].EvCount
	})
	for _, d := range tr.Data[funcLimit:] {
		tr.EvCountSkipped += d.EvCount
		delete(tr.tdFromCs, d.cs)
	}
	tr.Data = tr.Data[:funcLimit]
}

func (tr *TopRecord) MarshalJSON() ([]byte, error) {
	return json.Marshal(map[string]interface{}{
		"ts":              fmt.Sprintf("%d", tr.TS.Unix()),
		"fqdn":            tr.FQDN,
		"cgrp":            tr.Name,
		"evcount":         tr.EvCount,
		"evcount_skipped": tr.EvCountSkipped,
		"data":            tr.Data,
	})
}

type TopRecords struct {
	mux        sync.Mutex
	trFromName map[string]*TopRecord
}

func NewTopRecords() *TopRecords {
	return &TopRecords{
		trFromName: map[string]*TopRecord{},
	}
}

func (trs *TopRecords) update(rs []*TopRecord) {
	trs.mux.Lock()
	defer trs.mux.Unlock()
	for _, r := range rs {
		if tr, ok := trs.trFromName[r.Name]; ok {
			tr.update(r)
		} else {
			trs.trFromName[r.Name] = r
		}
	}
}

func recordsRead(ts time.Time, fqdn string, cgNames []string, pi *podsinfo.PodsInfo, buf *bytes.Buffer) ([]*TopRecord, error) {
	var (
		tr      *TopRecord
		rs      = []*TopRecord{}
		cgName  string
		podName string
	)

	scan := bufio.NewScanner(buf)
	for cgNameIdx := 0; scan.Scan(); {
		if err := scan.Err(); err != nil {
			return nil, err
		}
		s := scan.Text()
		if strings.HasPrefix(s, "# Event count") {
			l := strings.Fields(s)
			if cgNameIdx >= len(cgNames) {
				break
			}
			cgName = cgNames[cgNameIdx]
			cgNameIdx++
			if len(l) < 4 {
				return nil, fmt.Errorf("unsupported format: %s", s)
			}
			evCount, err := strconv.ParseUint(l[4], 10, 64)
			if err != nil {
				return nil, fmt.Errorf("unsupported format: %s", s)
			} else if evCount == 0 {
				continue
			}
			if cgName == "kernel" {
				podName = cgName
			} else {
				pod := pi.FromCgName(cgName)
				if pod == nil {
					podName = ""
					continue
				}
				podName = pod.Name
			}
			tr = NewTopRecord(
				ts,
				fqdn,
				podName,
				evCount,
			)
			rs = append(rs, tr)
			continue
		}
		l := strings.Split(s, FieldsDelim)
		if podName == "" {
			continue
		}
		td, err := TopDataParse(s, l)
		if err != nil {
			continue
		}
		// skip user funcs for host
		// skip idle funcs
		if (cgName == "kernel" && td.st == UserSpace) ||
			(td.st == KernelSpace && td.Sym == "idle") {
			continue
		}
		tr.add(td)
	}
	return rs, nil
}

func symIsIdle(s string) bool {
	for _, i := range IdleFuncs {
		if strings.Contains(i, s) {
			return true
		}
	}
	return false
}

func checksum(s string) string {
	cs := sha1.New()
	cs.Write([]byte(s))
	return fmt.Sprintf("%x", cs.Sum(nil))
}

func minInt(x, y int) int {
	if x < y {
		return x
	}
	return y
}
