package perf

import (
	"bytes"
	"strings"
	"testing"
	"time"

	"a.yandex-team.ru/infra/rsm/perfmanager/internal/podsinfo"
	"github.com/stretchr/testify/assert"
)

func TestTopDataParse(t *testing.T) {
	var (
		tests = []struct {
			actual   string
			expected *TopData
			err      bool
		}{
			{
				"209770704 ##rtc-sla-tentacl ##rtc-sla-tentacles-agent ##[.] _PyEval_EvalFrameDefault",
				&TopData{
					EvCount: 209770704,
					DSO:     "rtc-sla-tentacles-agent",
					Comm:    "rtc-sla-tentacl",
					Sym:     "_PyEval_EvalFrameDefault",
					cs:      checksum("rtc-sla-tentacles-agent" + "rtc-sla-tentacl" + "_PyEval_EvalFrameDefault"),
					st:      UserSpace,
				},
				false,
			},
			{
				"209770705 ##base_mapper ##yabs-server ##[.] std::__y1::__sort<void SortBy<(anonymous namespace)::TWordNetMerger::TLink*, (anonymous namespace)::TWordNetMerger::123qweasdzxc",
				&TopData{
					EvCount: 209770705,
					DSO:     "yabs-server",
					Comm:    "base_mapper",
					Sym:     "std::__y1::__sort<void SortBy<(anonymous namespace)::TWordNetMerger::TLink*, (anonymous namespace)::TWordNetMerger::123qweasdzxc",
					cs:      checksum("yabs-server" + "base_mapper" + "std::__y1::__sort<void SortBy<(anonymous namespace)::TWordNetMerger::TLink*, (anonymous namespace)::TWordNetMerger::123qweasdzxc"),
					st:      UserSpace,
				},
				false,
			},
			{
				"209770701 ##base_mapper ##yabs-server ##[.] xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx",
				&TopData{
					EvCount: 209770701,
					DSO:     "yabs-server",
					Comm:    "base_mapper",
					Sym:     "xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx",
					cs:      checksum("yabs-server" + "base_mapper" + "xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx"),
					st:      UserSpace,
				},
				false,
			},
			{
				"XXX ##base_mapper ##yabs-server ##[.] std::__y1::",
				nil,
				true,
			},
			{
				"209770703 ##base_mapper ##yabs-server ##[.]  std;::__y1::",
				&TopData{
					EvCount: 209770703,
					DSO:     "yabs-server",
					Comm:    "base_mapper",
					Sym:     "std;::__y1::",
					cs:      checksum("yabs-server" + "base_mapper" + "std;::__y1::"),
					st:      UserSpace,
				},
				false,
			},
			{
				"209770706 ##rcu_sched ##[kernel.kallsyms] ##[k] dyntick_save_progress_counter",
				&TopData{
					EvCount: 209770706,
					DSO:     "kernel",
					Comm:    "",
					Sym:     "dyntick_save_progress_counter",
					cs:      checksum("kernel" + "dyntick_save_progress_counter"),
					st:      KernelSpace,
				},
				false,
			},
		}
	)

	for _, test := range tests {
		td, err := TopDataParse(test.actual, strings.Split(test.actual, FieldsDelim))
		if test.err {
			assert.Error(t, err)
		} else {
			assert.NoError(t, err)
		}
		assert.Equal(t, test.expected, td)
	}
}

func TestTopRecord(t *testing.T) {
	type test struct {
		id       uint
		actual   *TopRecord
		expected *TopRecord
	}

	var (
		FuncLimit = 2
		tsNow1    = time.Now()
		tsNow2    = time.Now()
		tests     = []test{}
	)

	// test 1 (update)
	tr1 := NewTopRecord(tsNow1, "", "", 0)
	tr2 := NewTopRecord(tsNow2, "", "", 0)
	tr1.update(tr2)
	tr3 := NewTopRecord(tsNow2, "", "", 0)
	tests = append(tests, test{1, tr1, tr3})

	// test 2 (add and trim)
	tr1 = NewTopRecord(tsNow1, "", "", 0)
	tr2 = NewTopRecord(tsNow1, "", "", 0)
	tr1.update(tr2)
	tr3 = NewTopRecord(tsNow1, "", "", 0)
	tests = append(tests, test{2, tr1, tr3})

	// test 3 (add and trim)
	tr1 = NewTopRecord(tsNow1, "", "", 6)
	tr1.add(NewTopData(1, "comm1", "dso1", "func1", UserSpace))
	tr1.add(NewTopData(2, "comm2", "dso2", "func2", UserSpace))
	tr1.add(NewTopData(3, "comm3", "dso3", "func3", UserSpace))
	tr2 = NewTopRecord(tsNow2, "", "", 4)
	tr2.add(NewTopData(1, "comm1", "dso1", "func1", UserSpace))
	tr2.add(NewTopData(2, "comm2", "dso2", "func2", UserSpace))
	tr2.add(NewTopData(1, "comm1", "dso1", "func1", UserSpace))
	tr1.update(tr2)
	tr1.trim(FuncLimit)
	tr3 = NewTopRecord(tsNow2, "", "", 10)
	tr3.add(NewTopData(3, "comm1", "dso1", "func1", UserSpace))
	tr3.add(NewTopData(4, "comm2", "dso2", "func2", UserSpace))
	tr3.add(NewTopData(0, "comm3", "dso3", "func3", UserSpace))
	tr3.EvCountSkipped = 3
	tr3.trim(FuncLimit)
	tests = append(tests, test{2, tr1, tr3})

	for _, tt := range tests {
		assert.Equal(t, tt.expected, tt.actual)
	}
}

func TestCollect(t *testing.T) {
	buf1 := bytes.NewBufferString(`
# To display the perf.data header info, please use --header/--header-only options.
#
#
# Total Lost Samples: 0
#
# Samples: 0  of event 'cycles'
# Event count (approx.): 7
#
#       Period  Command  Shared Object  Symbol
# ............  .......  .............  ......
#
		4## comm14 ##[JIT XXX] ##[.] 0x000001
		2## comm15 ##9VT2LOtc.vbt ##[.] 0x000002
		1## comm15 ##libbrotli.so-1653326403080101770755514473688.tmp ##[.] 0x000003


# Samples: 20  of event 'cycles'
# Event count (approx.): 65
#
#       Period  Command          Shared Object            Symbol
# ............  ...............  .......................  ..........................
#
		1 ##comm11 ##[kernel.kallsyms] ##[k] __fget_light
			10		##comm11	 ##dso11	 ##[.]   func11	  
		20 ##comm12 ##dso12## [.] func12
		30## comm13## dso13 ##[.] func13
		1## comm12## [kernel.kallsyms] ##[k] __intel_pmu_enable_all
		3## comm13 ##[kernel.kallsyms] ##[k] __intel_pmu_enable_all


# Samples: 1K of event 'cycles'
# Event count (approx.): 230
#
#       Period  Command          Shared Object         Symbol
# ............  ...............  ....................  .......................................................................................................................................................................................................................................................................
#
		60## comm21 ##dso21 ##[k] func21
    55 ##yandex-hbf-agen ##hbf-agent ##[.] PyEval_EvalFrameEx
    50 ##comm22 ##dso22 ##[k] func22
    45 ##comm23 ##dso23 ##[k] func23
    15 ##bash ##libc-2.23.so ##[.] __mbrtowc
    5 ##comm24 ##dso22 ##[k] func22
	`)
	buf2 := bytes.NewBufferString(`
# To display the perf.data header info, please use --header/--header-only options.
#
#
# Total Lost Samples: 0
#
# Samples: 20  of event 'cycles'
# Event count (approx.): 289
#
#       Period  Command          Shared Object            Symbol
# ............  ...............  .......................  ..........................
#
		70 ##comm1 ##dso1 ##[.] func1
    30 ##comm1 ##[kernel.kallsyms] ##[k] __fget_light
    60 ##comm2 ##dso2 ##[.] func2
    80 ##comm3 ##dso3 ##[.] func3
    45 ##comm4 ##dso4 ##[.] func4
		4## comm14 ##[JIT 123] ##[.] 0x000005


# Samples: 0  of event 'cycles'
# Event count (approx.): 0
#
#       Period  Command  Shared Object  Symbol
# ............  .......  .............  ......
#


# Samples: 1K of event 'cycles'
# Event count (approx.): 250
#
#       Period  Command          Shared Object         Symbol
# ............  ...............  ....................  .......................................................................................................................................................................................................................................................................
#
			60		 ##comm21   ##dso21	 ##[k] func21						
    55 ##yandex-hbf-agen ##hbf-agent ##[.] PyEval_EvalFrameEx
    50 ##comm22 ##dso22 ##[k] func22
    45 ##comm222 ##dso222 ##[k] func23
    40 ##comm222 ##dso222 ##[k] func24
	`)
	ts1 := time.Now()
	ts2 := time.Now()
	podsNames := []string{"pod1", "pod2"}
	ctNames := []string{"ISS-AGENT--1", "ISS-AGENT--2"}
	cgNames := []string{podsinfo.CgrpPrfxPorto + ctNames[0], podsinfo.CgrpPrfxPorto + ctNames[1]}
	trs := NewTopRecords()

	pod1 := podsinfo.NewPod(ctNames[0])
	pod1.Name = podsNames[0]
	assert.Equal(t, pod1.Name, podsNames[0])

	pod2 := podsinfo.NewPod(ctNames[1])
	pod2.Name = podsNames[1]
	assert.Equal(t, pod2.Name, podsNames[1])

	pi := &podsinfo.PodsInfo{}
	pi.SetCache([]*podsinfo.Pod{pod1, pod2})

	assert.Equal(t, cgNames, pi.CgNames())
	assert.Equal(t, ctNames, pi.CtNames())
	assert.Equal(t, pi.FromCgName(cgNames[0]).Name, podsNames[0])
	assert.Equal(t, pi.FromCgName(cgNames[1]).Name, podsNames[1])

	cgNames = append(cgNames, "kernel")
	rs, err := recordsRead(ts1, "test", cgNames, pi, buf1)
	assert.NoError(t, err)
	trs.update(rs)

	rs, err = recordsRead(ts2, "test", cgNames, pi, buf2)
	assert.NoError(t, err)
	trs.update(rs)

	tr1 := NewTopRecord(ts2, "test", podsNames[0], 296)
	tr1.add(NewTopData(8, "JIT", "comm14", "unknown", UserSpace))
	tr1.add(NewTopData(3, "unknown", "comm15", "unknown", UserSpace))
	tr1.add(NewTopData(70, "dso1", "comm1", "func1", UserSpace))
	tr1.add(NewTopData(30, "kernel", "", "__fget_light", KernelSpace))
	tr1.add(NewTopData(60, "dso2", "comm2", "func2", UserSpace))
	tr1.add(NewTopData(80, "dso3", "comm3", "func3", UserSpace))
	tr1.add(NewTopData(45, "dso4", "comm4", "func4", UserSpace))

	tr2 := NewTopRecord(ts1, "test", podsNames[1], 65)
	tr2.add(NewTopData(1, "kernel", "", "__fget_light", KernelSpace))
	tr2.add(NewTopData(10, "dso11", "comm11", "func11", UserSpace))
	tr2.add(NewTopData(20, "dso12", "comm12", "func12", UserSpace))
	tr2.add(NewTopData(30, "dso13", "comm13", "func13", UserSpace))
	tr2.add(NewTopData(4, "kernel", "", "__intel_pmu_enable_all", KernelSpace))

	tr3 := NewTopRecord(ts2, "test", cgNames[2], 480)
	tr3.add(NewTopData(120, "kernel", "", "func21", KernelSpace))
	tr3.add(NewTopData(105, "kernel", "", "func22", KernelSpace))
	tr3.add(NewTopData(90, "kernel", "", "func23", KernelSpace))
	tr3.add(NewTopData(40, "kernel", "", "func24", KernelSpace))

	trsExpected := NewTopRecords()
	trsExpected.update([]*TopRecord{tr1, tr2, tr3})
	assert.Equal(t, trsExpected, trs)
}
