package podsinfo

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"regexp"
	"strings"
	"sync"
	"time"

	porto_api "a.yandex-team.ru/infra/porto/api_go"
	"a.yandex-team.ru/library/go/core/log/zap"
)

const (
	PortoStatsUpdateIval = 15 * time.Second
	IssTimeout           = 2 * time.Second
	IssURLPods           = "http://localhost:25536/pods/info/"
	CgrpPrfxPorto        = "porto%"
	CgrpPrfxIss          = "ISS-AGENT--"
)

var (
	// ISS-AGENT--21587
	PodRgxp1 = regexp.MustCompile(CgrpPrfxIss + `[0-9]+$`)
	// ISS-AGENT--15997_vla_psi_lite_arnold_BSOTAqXPORJ
	PodRgxp2 = regexp.MustCompile(CgrpPrfxIss + `[0-9]+_(.*)_[a-zA-Z0-9]+$`)
	// 21587_vla_jupiter_remote_storage_gh15BMEilLK
	SubPodRgxp = regexp.MustCompile(`[0-9]+_(.*)_[a-zA-Z0-9]+$`)
)

type issPodInfoResp struct {
	SetID string `json:"pod_set_id"`
}

type PodsInfo struct {
	l      *zap.Logger
	issC   *http.Client
	portoC porto_api.PortoAPI
	mux    sync.Mutex
	cache  []*Pod
}

func New(l *zap.Logger) (*PodsInfo, error) {
	portoC, err := porto_api.Dial()
	if err != nil {
		return nil, err
	}
	return &PodsInfo{
		l:      l,
		issC:   &http.Client{Timeout: IssTimeout},
		portoC: portoC,
		cache:  []*Pod{},
	}, nil
}

func (pi *PodsInfo) CtNames() (result []string) {
	pi.mux.Lock()
	defer pi.mux.Unlock()
	for _, p := range pi.cache {
		result = append(result, p.CtName)
	}
	return
}

func (pi *PodsInfo) FromCtName(n string) (p *Pod) {
	pi.mux.Lock()
	defer pi.mux.Unlock()
	for _, p = range pi.cache {
		if p.CtName == n {
			return
		}
	}
	return nil
}

func (pi *PodsInfo) CgNames() (result []string) {
	pi.mux.Lock()
	defer pi.mux.Unlock()
	for _, p := range pi.cache {
		result = append(result, p.CgName)
	}
	return
}

func (pi *PodsInfo) FromCgName(n string) (p *Pod) {
	pi.mux.Lock()
	defer pi.mux.Unlock()
	for _, p = range pi.cache {
		if p.CgName == n {
			return
		}
	}
	return nil
}

func (pi *PodsInfo) RunUpdateCache(ctx context.Context) (err error) {
	tick := time.NewTicker(PortoStatsUpdateIval)
	for {
		select {
		case ts := <-tick.C:
			if err := pi.updatePods(); err != nil {
				pi.l.Errorf("%s\n", err)
			}
			if err := pi.updateStats(ts); err != nil {
				pi.l.Errorf("%s\n", err)
			}
		case <-ctx.Done():
			return ctx.Err()
		}
	}
}

func (pi *PodsInfo) updatePods() error {
	var (
		pod      *Pod
		ctPrev   string
		cacheNew = []*Pod{}
	)

	pi.l.Debug("Start updating pods list")
	cts, err := pi.portoC.ListContainers(CgrpPrfxIss + "***")
	if err != nil {
		return err
	}
	for _, ct := range cts {
		ctL := strings.Split(ct, "/")
		ctMeta := ctL[0]
		if len(ctL) != 2 || strings.Compare(ctMeta, ctPrev) == 0 {
			continue
		}
		ctPrev = ctMeta

		// TODO: KERNEL-872
		if strings.Contains(ct, "_yt_") && !strings.Contains(ct, "_yt_hume_dat_nodes_") {
			pi.l.Debugf("Skip YT container: %s", ct)
			continue
		}

		// does exist in cache?
		if pod = pi.FromCtName(ctMeta); pod != nil {
			cacheNew = append(cacheNew, pod)
			continue
		} else {
			pod = NewPod(ctMeta)
		}

		// parse ctName to podName
		// ISS-AGENT--21587/21587_vla_jupiter_remote_storage_gh15BMEilLK -> vla_jupiter_remote_storage
		if len(ctL) > 1 && PodRgxp1.MatchString(ctMeta) {
			if r := SubPodRgxp.FindStringSubmatch(ctL[1]); len(r) > 0 {
				pod.Name = r[1]
			}
			// ISS-AGENT--15997_vla_psi_lite_arnold_BSOTAqXPORJ -> vla_psi_lite_arnold
		} else if r := PodRgxp2.FindStringSubmatch(ctMeta); len(r) > 0 {
			pod.Name = r[1]
			// ISS-AGENT--vev4zqa4odmewbwp -> maps-auto-remote-tasks-manager-stable
		} else {
			data, err := pi.getPodInfo(strings.TrimPrefix(ctMeta, CgrpPrfxIss))
			if err != nil {
				continue
			}
			pod.Name = data.SetID
		}

		if pod.Name != "" {
			cacheNew = append(cacheNew, pod)
		}
	}
	pi.SetCache(cacheNew)
	return nil
}

func (pi *PodsInfo) updateStats(ts time.Time) (err error) {
	pi.l.Debug("Start updating pods stats")
	stats, err := pi.portoC.GetProperties(pi.CtNames(), StatNames)
	if err != nil {
		return
	}
	for ct, data := range stats {
		if pod := pi.FromCtName(ct); pod != nil {
			if err = pod.updateStats(ts, data); err != nil {
				return
			}
			pi.l.Debugf("%+v", pod)
		}
	}
	return
}

func (pi *PodsInfo) SetCache(m []*Pod) {
	pi.mux.Lock()
	defer pi.mux.Unlock()
	pi.cache = m
}

func (pi *PodsInfo) getPodInfo(n string) (*issPodInfoResp, error) {
	resp, err := pi.issC.Get(IssURLPods + strings.TrimPrefix(n, CgrpPrfxIss))
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	body, _ := ioutil.ReadAll(resp.Body)
	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("http error, code: %d, err: %s", resp.StatusCode, string(body))
	}
	info := &issPodInfoResp{}
	err = json.Unmarshal(body, info)
	return info, err
}
