package podsinfo

import (
	"fmt"
	"strconv"
	"strings"
	"sync"
	"time"
)

const (
	StatNameNetRxBytes   = "net_rx_bytes"
	StatNameNetTxBytes   = "net_tx_bytes"
	StatNameNetRxPkts    = "net_rx_packets"
	StatNameNetTxPkts    = "net_tx_packets"
	StatNameIOReadBytes  = "io_read"
	StatNameIOWriteBytes = "io_write"
	StatNameCPUUsage     = "cpu_usage"
	StatCacheTimeout     = time.Minute * 15
)

var (
	StatNameToFunc = map[string]StatParseFunc{
		StatNameNetRxBytes:   ParseNet,
		StatNameNetTxBytes:   ParseNet,
		StatNameNetRxPkts:    ParseNet,
		StatNameNetTxPkts:    ParseNet,
		StatNameIOReadBytes:  ParseIO,
		StatNameIOWriteBytes: ParseIO,
		StatNameCPUUsage:     ParseUint,
	}
	StatNames = []string{
		StatNameNetRxBytes,
		StatNameNetTxBytes,
		StatNameNetRxPkts,
		StatNameNetTxPkts,
		StatNameIOReadBytes,
		StatNameIOWriteBytes,
		StatNameCPUUsage,
	}
)

type StatParseFunc func(s string) (uint64, error)

type Backet struct {
	ts  time.Time
	val uint64
}

type Stat struct {
	pf  StatParseFunc
	mux sync.Mutex
	b   [2]*Backet
}

func NewStat(f StatParseFunc) *Stat {
	return &Stat{
		pf: f,
		b:  [2]*Backet{&Backet{}, &Backet{}},
	}
}

func (s *Stat) UpdateFromStr(newTS time.Time, data string) (err error) {
	newVal, err := s.pf(data)
	if err != nil {
		return
	}
	s.mux.Lock()
	defer s.mux.Unlock()
	s.b[0].ts = s.b[1].ts
	s.b[0].val = s.b[1].val
	s.b[1].ts = newTS
	s.b[1].val = newVal
	return
}

func (s *Stat) GetPerSec() uint64 {
	s.mux.Lock()
	defer s.mux.Unlock()
	prevTS := s.b[0].ts
	prevVal := s.b[0].val
	lastTS := s.b[1].ts
	lastVal := s.b[1].val
	if prevTS.Before(lastTS.Add(-StatCacheTimeout)) {
		return 0
	}
	secs := uint64(lastTS.Sub(prevTS).Seconds())
	if secs == 0 {
		secs = 1
	}
	return (lastVal - prevVal) / secs
}

func (s *Stat) GetAbs() uint64 {
	s.mux.Lock()
	defer s.mux.Unlock()
	return s.b[1].val
}

// Latency: 3.4K; Uplink: 884785375486312; group default: 884785375486312; ip6tnl0: 0; veth: 884785375486312
func ParseNet(s string) (c uint64, err error) {
	if s == "" {
		return
	}
	s, err = ParsePortoProps(s, "Uplink")
	if err != nil {
		return
	}
	c, err = strconv.ParseUint(s, 10, 64)
	if err != nil {
		return
	}
	return
}

// hw: 279667039096832; sda: 35062034997248; sdb: 35348836864000;
func ParseIO(s string) (c uint64, err error) {
	if s == "" {
		return
	}
	s, err = ParsePortoProps(s, "hw")
	if err != nil {
		return
	}
	c, err = strconv.ParseUint(s, 10, 64)
	if err != nil {
		return
	}
	return
}

// 1750334174
func ParseUint(s string) (c uint64, err error) {
	if s == "" {
		return
	}
	c, err = strconv.ParseUint(strings.Trim(s, " "), 10, 64)
	if err != nil {
		return
	}
	return
}

// hw: 279667039096832; sda: 35062034997248; sdb: 35348836864000;
func ParsePortoProps(s, n string) (string, error) {
	for _, v := range strings.Split(s, ";") {
		sl := strings.Split(v, ":")
		if len(sl) != 2 {
			return "", fmt.Errorf("parsing porto properties: %s", s)
		}
		if strings.Trim(sl[0], " ") != n {
			continue
		}
		return strings.Trim(sl[1], " "), nil
	}
	return "", fmt.Errorf("%s property not found", n)
}
