
"""
на вход принимаем список всех сигрупп (то есть нянесервисов или деплой энжинов)
возвращаем мапу "сигруппа-абц" (верхнеуровневый и обычный)
"""

import requests
from collections import defaultdict
import os
import logging
import json
import threading
from yp.client import YpClient, find_token
from yp.common import YP_CLUSTERS
import yt.wrapper
import nirvana.job_context as nv


from infra.rsm.perfmanager.scripts.resolve_cgrps.abc_api_wrapper import AbcApi
from infra.rsm.perfmanager.scripts.resolve_cgrps.utils import get_table_name, split_chunks, extract_top_parents, extract_gencfg_group_itag_position, clean_abc_id, DEFAULT_ABC_SERVICE


def prepare_gencfg_mapping():
    gencfg_table = get_table_name()

    gencfg_abc_map = {}

    for record in yt.wrapper.read_table(gencfg_table):
        gencfg_abc_map[record["group_name"]] = record["account_id"]

    return gencfg_abc_map


def extract_already_resolved():

    resolved_cgrps_path = "//home/runtimecloud/research/KERNEL-691/cgrps_resolved"
    already_resolved = defaultdict(dict)

    try:
        for record in yt.wrapper.read_table(resolved_cgrps_path):
            already_resolved[record["cgrp"]] = {
                "service_name": record["service_name"],
                "top_paret": record["top_paret"],
                "pretop_parent": record["pretop_parent"],
                "abc_slug": record["abc_slug"],
                "account_id": str(record["account_id"])
            }
    except:
        return {}

    return already_resolved


def resolve_gencfg_service(servicename, gencfg_mapping, abc_api):
    nanny_token = os.environ.get("NANNY_TOKEN")
    headers = {"Authorization": "OAuth {}".format(nanny_token)}

    url = "https://nanny.yandex-team.ru/v2/services/{0}/current_state/instances/".format(servicename)
    data = requests.get(url, headers=headers).json().get("result", [])

    if len(data) > 0:

        itags = data[0].get("itags", [])
        if len(itags) > 0:

            group_index = extract_gencfg_group_itag_position(itags)

            if group_index is None:
                return

            gencfg_group = itags[group_index]
            try:
                mpping = extract_abc_params(gencfg_mapping[gencfg_group], abc_api)
            except KeyError:
                return
            return mpping

        else:
            return
    else:
        return


def parallelized_nanny_resolving(unresolved, abc_api):

    splitted_services = list(split_chunks(unresolved, int(len(unresolved) / 5)))
    gencfg_mapping = prepare_gencfg_mapping()
    result, threads = {}, []

    for i in range(len(splitted_services)):
        t = threading.Thread(target=get_nanny_abc, args=(splitted_services[i], abc_api, gencfg_mapping, result))
        t.start()
        threads.append(t)

    for thread in threads:
        thread.join()

    return result


def unparallelized_nanny_resolving(unresolved, abc_api):
    gencfg_mapping = prepare_gencfg_mapping()
    result = {}

    get_nanny_abc(unresolved, abc_api, gencfg_mapping, result)

    return result


def prepare_podset_account_map():

    YP_CLUSTERS.remove("pre")
    YP_CLUSTERS.remove("sas-test")

    mapping = {}

    for dc in YP_CLUSTERS:
        client = YpClient(dc, config=dict(token=find_token()))

        podset_accounts = client.select_objects("pod_set",
                                                selectors=["/meta/id", "/meta/account_id"],
                                                enable_structured_response=True
                                                )["results"]

        for record in podset_accounts:
            mapping[record[0]["value"]] = record[1]["value"]

    return mapping


def get_nanny_abc(servicenames, abc_api, gencfg_mapping, result):

    nanny_token = os.environ.get("NANNY_TOKEN")
    headers = {"Authorization": "OAuth {}".format(nanny_token)}

    for servicename in servicenames:
        url = "https://nanny.yandex-team.ru/v2/services/{0}/info_attrs/".format(servicename)
        abc_id = str(requests.get(url, headers=headers).json().get("content", {}).get("abc_group", DEFAULT_ABC_SERVICE))
        if abc_id == "0":
            gencfg_resolution = resolve_gencfg_service(servicename, gencfg_mapping, abc_api)
            if gencfg_resolution:
                result[servicename] = gencfg_resolution
                continue

            abc_id = DEFAULT_ABC_SERVICE

        resolved_name = extract_abc_params(abc_id, abc_api)

        if not resolved_name:
            result[servicename] = {
                "service_name": DEFAULT_ABC_SERVICE, "top_paret": DEFAULT_ABC_SERVICE,
                "pretop_parent": DEFAULT_ABC_SERVICE, "abc_slug": DEFAULT_ABC_SERVICE,
                "account_id": DEFAULT_ABC_SERVICE
            }

        else:
            result[servicename] = resolved_name


def extract_abc_params(abc_id, abc_api):
    technical_abc = False

    if not abc_id or abc_id == DEFAULT_ABC_SERVICE:
        return
    else:
        try:
            abc_integer = int(clean_abc_id(abc_id))
        except ValueError:
            technical_abc = True
            abc_integer = None

        if technical_abc:
            return {
                "service_name": abc_id, "top_paret": abc_id, "pretop_parent": abc_id, "abc_slug": abc_id,
                "account_id": abc_id
            }

        service_name = abc_api.get_service_name(abc_api.get_service_slug(service_id=abc_integer))
        service_parents = abc_api.get_service_parents_names(service_id=abc_integer)
        service_slug = abc_api.get_service_slug(service_id=abc_integer)
        account_id = str(abc_api.get_service_id(service_slug))

        highest_parent, previous_parent = extract_top_parents(service_name, service_parents)

        return {
            "service_name": service_name, "top_paret": highest_parent, "pretop_parent": previous_parent,
            "abc_slug": service_slug, "account_id": account_id
        }


def resolve_cgrps(cgrps, abc_api):

    pod_set_map = prepare_podset_account_map()
    unresolved, resolved = [], {}

    for cgrp in cgrps:
        abc_id = pod_set_map.get(cgrp)

        abc_params = extract_abc_params(abc_id, abc_api)

        if not abc_params:
            unresolved.append(cgrp)
        else:
            resolved[cgrp] = abc_params

    return resolved, unresolved


if __name__ == '__main__':
    abcapi = AbcApi(token=os.environ.get("ABC_TOKEN"))
    logger = logging

    while True:
        attempts = 1
        try:
            abcapi.load_abc_service_tree()
            break
        except requests.exceptions.JSONDecodeError:
            logging.info("Number of unsuccessfull atempts to read data from abc api: {0}".format(str(attempts)))

    yt.wrapper.config.set_proxy("hahn")

    job_context = nv.context()
    inputs, outputs = job_context.get_inputs(), job_context.get_outputs()

    data = json.load(open(inputs.get(os.environ.get("INPUT_FILE"))))
    already_resolved_mapping = extract_already_resolved()

    cgrps = []

    for record in data:
        if record["cgrp"] not in already_resolved_mapping:
            cgrps.append(record["cgrp"])

    resolved_deploy, unresolved = resolve_cgrps(cgrps, abcapi)

    resolved_nanny = parallelized_nanny_resolving(unresolved, abcapi)

    filename = os.environ.get("OUTPUT_FILE")
    with open(outputs.get(filename), "w") as write_file:

        data, result = {**resolved_deploy, **resolved_nanny, **already_resolved_mapping}, []

        for cgrp, cgrp_data in data.items():
            cgrp_data["cgrp"] = cgrp
            result.append(cgrp_data)

        json.dump(result, write_file)
