package cmd

import (
	"fmt"
	"os"
	"syscall"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/infra/rsm/sysconf/internal/atboot"
	"a.yandex-team.ru/infra/rsm/sysconf/internal/plugin"
)

const (
	timeFmt = "02-01-2006 15:04:05"
)

func executeRun(auto, force bool) {
	var (
		ph  atboot.Phase = atboot.Live("live-default")
		err error
	)

	if force {
		ph = atboot.Force("force-flag")
	} else if auto {
		ph, err = atboot.GetPhase()
		if err != nil {
			fmt.Printf("Failed to determine phase: %s\n", err)
			os.Exit(1)
		}
	}

	defer func() {
		if err := ph.Done(); err != nil {
			fmt.Printf("Failed to mark phase '%s' as done: %s\n", ph, err)
		}
	}()

	fmt.Printf("### %s (phase=%s) ###\n", time.Now().Format(timeFmt), ph)
	for _, p := range plugin.List() {
		err := p.IsApplicable()
		fmt.Printf("%s: %s\n", p.Name(), err.Status)
		if err.Status == plugin.StatusFail {
			fmt.Println(err.Err)
		}
		if err.Status != plugin.StatusEnable {
			continue
		}
		for _, err := range p.Enable(ph.Force()) {
			fmt.Println("", err.String())
		}
	}
}

func redirectStdout(path string) error {
	f, err := os.OpenFile(path, os.O_RDWR|os.O_CREATE|os.O_APPEND, 0644)
	if err != nil {
		return err
	}
	// Dup newly opened file to stdout/stderr file descriptors
	if err := syscall.Dup3(int(f.Fd()), 1, 0); err != nil {
		return fmt.Errorf("failed to redirect stdout: %w", err)
	}
	if err := syscall.Dup3(int(f.Fd()), 2, 0); err != nil {
		return fmt.Errorf("failed to redirect stderr: %w", err)
	}
	return nil
}

func init() {
	var (
		auto bool
		out  string
	)

	runCmd := &cobra.Command{
		Use:   "run",
		Short: "Apply applicable plugins",
		Run: func(cmd *cobra.Command, args []string) {
			if len(out) != 0 {
				if err := redirectStdout(out); err != nil {
					fmt.Printf("Failed to redirect stdout: %s\n", err)
					os.Exit(1)
				}
			}
			executeRun(auto, force)
		},
	}

	runCmd.Flags().BoolVarP(&auto, "auto", "", false,
		"auto detect if force run required")
	runCmd.Flags().StringVarP(&out, "out-file", "o", "",
		"redirect stdout/stderr to file (default: no redirection)")
	rootCmd.AddCommand(runCmd)
}
