package atboot

import (
	"os"
	"time"

	"a.yandex-team.ru/infra/rsm/sysconf/internal/sysinfo"
)

const (
	// How long in uptime we assume that disruptive changes are OK
	atBootThreshold = 10000 * time.Hour
	// Marker file analogous to
	// https://github.com/svagner/vixie-cron/blob/13d13f0b38d73e7a415caa77b5821be92dd16f70/cron.c#L287
	atBootFile = "/var/run/yandex-sysconf.reboot"
)

type Phase interface {
	Force() bool
	Done() error
}

type AtBoot string

func (boot AtBoot) Force() bool {
	return true
}

func (boot AtBoot) Done() error {
	_, err := os.Create(atBootFile)
	return err
}

type Live string

func (l Live) Force() bool {
	return false
}

func (l Live) Done() error {
	return nil
}

type Force string

func (f Force) Force() bool {
	return true
}

func (f Force) Done() error {
	return nil
}

// GetPhase tries to infer host phase by looking at flag file
// or host uptime.
func GetPhase() (Phase, error) {
	_, err := os.Stat(atBootFile)
	if err == nil {
		return Live("live-marker-triggered"), nil
	}
	if !os.IsNotExist(err) {
		return nil, err
	}
	// File does not exists, but maybe we're started too late
	// or just migrated to this scheme - check host uptime
	uptime, err := sysinfo.Uptime()
	if err != nil {
		return nil, err
	}
	// Oh no, too late, assume live phase.
	if uptime > atBootThreshold {
		return Live("live-uptime-triggered"), nil
	}
	return AtBoot("at-boot"), nil
}
