package cpufreq

import (
	"fmt"
	"io/ioutil"
	"path/filepath"
	"regexp"

	"a.yandex-team.ru/infra/rsm/sysconf/internal"
	"a.yandex-team.ru/infra/rsm/sysconf/internal/plugin"
)

const (
	governorFileRgxp    = "/sys/devices/system/cpu/cpu*/cpufreq/scaling_governor"
	governorFileContent = "performance"
	cpuInfoPath         = "/proc/cpuinfo"
)

var (
	// RTCNETWORK-600
	cpuExcludeModelsRgxp = regexp.MustCompile(`model name.*(E5645).*`)
)

func init() {
	plugin.Register(&pl{})
}

type pl struct{}

func (p *pl) Name() string {
	return "cpufreq"
}

func (p *pl) Doc() string {
	return "https://st.yandex-team.ru/RTCNETWORK-564"
}

func (p *pl) IsApplicable() (st plugin.State) {
	data, err := ioutil.ReadFile(cpuInfoPath)
	if err != nil {
		st.Err = err
		st.Status = plugin.StatusFail
		return
	}
	if cpuExcludeModelsRgxp.Find(data) != nil {
		st.Status = plugin.StatusSkip
		return
	}
	st.Status = plugin.StatusEnable
	return
}

func (p *pl) Check() (sts plugin.States) {
	sts.Add(checkGovernorFeature())
	return
}

func (p *pl) Enable(force bool) (sts plugin.States) {
	var st plugin.State
	if st = checkGovernorFeature(); st.Status == plugin.StatusDiff {
		st = enableGovernorFeature()
	}
	sts.Add(st)
	return
}

func (p *pl) Disable(force bool) (sts plugin.States) {
	sts.Add(plugin.State{Name: "self", Status: plugin.StatusSkip, Err: plugin.ErrNotImpl})
	return
}

func enableGovernorFeature() (st plugin.State) {
	st = plugin.State{Name: "Governor", Status: plugin.StatusFail, Err: nil}
	files, err := filepath.Glob(governorFileRgxp)
	if err != nil {
		st.Err = err
		return
	}
	for _, f := range files {
		st.Err = ioutil.WriteFile(f, []byte(governorFileContent), 0)
		if st.Err != nil {
			return
		}
	}
	st.Status = plugin.StatusOk
	return
}

func checkGovernorFeature() (st plugin.State) {
	st = plugin.State{Name: "Governor", Status: plugin.StatusFail, Err: nil}
	files, err := filepath.Glob(governorFileRgxp)
	if err != nil {
		st.Err = err
		return
	}
	if len(files) == 0 {
		st.Err = fmt.Errorf("%s files does not exist", governorFileRgxp)
		return
	}
	for _, f := range files {
		st.Err = internal.IsFileContent(f, governorFileContent)
		if st.Err != nil {
			st.Status = plugin.StatusDiff
			return
		}
	}
	st.Status = plugin.StatusOk
	return
}
