package plugin

import (
	"errors"
	"fmt"
	"os"
)

type Status string

const (
	StatusOk     Status = "OK"
	StatusDiff   Status = "DIFF"
	StatusFail   Status = "FAIL"
	StatusSkip   Status = "SKIP"
	StatusEnable Status = "ENABLE"
)

var (
	ErrNotImpl   = errors.New("not implemented")
	ErrNeedForce = errors.New("need --force flag")
)

type State struct {
	Name   string
	Status Status
	Err    error
}

func (st *State) String() string {
	if st.Err != nil {
		return fmt.Sprintf("%-4s - %s: %v", st.Status, st.Name, st.Err)
	}
	return fmt.Sprintf("%-4s - %s", st.Status, st.Name)
}

type States []State

func (sts *States) Add(st State) State {
	*sts = append(*sts, st)
	return st
}

type Plugin interface {
	Name() string
	Doc() string
	Enable(bool) States
	Disable(bool) States
	Check() States
	IsApplicable() State
}

var registry map[string]Plugin

func init() {
	registry = map[string]Plugin{}
}

func Register(p Plugin) {
	if _, ok := registry[p.Name()]; ok {
		fmt.Printf("%s already exist", p.Name())
		os.Exit(1)
	}
	registry[p.Name()] = p
}

func List() []Plugin {
	var r []Plugin
	for _, p := range registry {
		r = append(r, p)
	}
	return r
}

func GetByName(n string) (Plugin, bool) {
	p, ok := registry[n]
	return p, ok
}
