package rdma

import (
	"fmt"
	"io/ioutil"
	"path"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/infra/rsm/sysconf/internal"
	"a.yandex-team.ru/infra/rsm/sysconf/internal/plugin"
	"a.yandex-team.ru/infra/rsm/sysconf/pkg/ethtool"
	"a.yandex-team.ru/infra/rsm/sysconf/pkg/mlx"
)

const (
	sysDevPath           = "/sys/bus/pci/devices"
	compatibleIfacesRgxp = "/sys/module/mlx_compat/holders/mlx5_core/drivers/*/*/net/eth[0-9]"
	tcFileContent        = "66"
	lossyRoceRegName     = "ROCE_ACCL"
	mlx4DevID            = "0x1015"
)

var (
	lossyRoceRegFields = []string{"roce_adp_retrans_en", "roce_tx_window_en", "roce_slow_restart_en"}
)

type ifaceDev struct {
	name string
	pci  string
}

func (i *ifaceDev) sysPath() string {
	return path.Join(sysDevPath, i.pci)
}

func (i *ifaceDev) tcFilePath() ([]string, error) {
	return filepath.Glob(path.Join(i.sysPath(), "infiniband/*/tc/*/traffic_class"))
}

type pl struct {
	ifaces []*ifaceDev
}

func New() (*pl, error) {
	files, err := filepath.Glob(compatibleIfacesRgxp)
	if err != nil {
		return nil, err
	}

	ifaces := []*ifaceDev{}
	for _, f := range files {
		p := strings.Split(f, "/")
		name := p[len(p)-1]
		pciSlot := p[len(p)-3]
		// exclude mlx4 devices
		if err := internal.IsNotFileContent(path.Join(sysDevPath, pciSlot, "/device"), mlx4DevID); err == nil {
			ifaces = append(ifaces, &ifaceDev{name, pciSlot})
		}
	}

	return &pl{
		ifaces: ifaces,
	}, nil
}

func init() {
	p, err := New()
	if err != nil {
		panic(err)
	}
	plugin.Register(p)
}

func (p *pl) Name() string {
	return "rdma"
}

func (p *pl) Doc() string {
	return "https://st.yandex-team.ru/RTCNETWORK-338"
}

func (p *pl) IsApplicable() (st plugin.State) {
	st.Status = plugin.StatusEnable
	if st.Err = internal.IsFileExists("/sys/module/mlx_compat/holders/mlx5_core"); st.Err != nil {
		st.Status = plugin.StatusSkip
		return
	}
	// I am so sorry, but it's better than to have two vyshivankas (see RTCNETWORK-766)
	if st.Err = internal.IsFileExists(mlx.CmdRegPath); st.Err != nil {
		st.Status = plugin.StatusSkip
		return
	}
	return
}

func (p *pl) Check() (sts plugin.States) {
	sts.Add(p.checkTCFeature())
	sts.Add(p.checkPauseFeature())
	sts.Add(p.checkLossyRoceFeature())
	return
}

func (p *pl) Enable(force bool) (sts plugin.States) {
	var st plugin.State
	if st = p.checkTCFeature(); st.Status == plugin.StatusDiff {
		st = p.enableTCFeature()
	}
	sts.Add(st)
	if st = p.checkPauseFeature(); st.Status == plugin.StatusDiff {
		st = p.enablePauseFeature()
	}
	sts.Add(st)
	if st := p.checkLossyRoceFeature(); st.Status == plugin.StatusDiff {
		st = p.enableLossyRoceFeature()
	}
	sts.Add(st)
	return
}

func (p *pl) Disable(force bool) (sts plugin.States) {
	sts.Add(plugin.State{Name: "self", Status: plugin.StatusSkip, Err: plugin.ErrNotImpl})
	return
}

func (p *pl) enableTCFeature() (st plugin.State) {
	st = plugin.State{Name: "TClass", Status: plugin.StatusFail, Err: nil}
	for _, iface := range p.ifaces {
		files, err := iface.tcFilePath()
		if err != nil {
			st.Err = err
			return
		}
		for _, f := range files {
			st.Err = ioutil.WriteFile(f, []byte(tcFileContent), 0)
			if st.Err != nil {
				return
			}
		}
	}
	st.Status = plugin.StatusOk
	return
}

func (p *pl) checkTCFeature() (st plugin.State) {
	st = plugin.State{Name: "TClass", Status: plugin.StatusFail, Err: nil}
	for _, iface := range p.ifaces {
		files, err := iface.tcFilePath()
		if err != nil {
			st.Err = err
			return
		}
		for _, f := range files {
			st.Err = internal.IsFileContent(f, "Global tclass="+tcFileContent)
			if st.Err != nil {
				st.Status = plugin.StatusDiff
				return
			}
		}
	}
	st.Status = plugin.StatusOk
	return
}

func (p *pl) enablePauseFeature() (st plugin.State) {
	st = plugin.State{Name: "PauseParam", Status: plugin.StatusFail, Err: nil}
	for _, iface := range p.ifaces {
		_, st.Err = ethtool.SetPauseParam(iface.name, ethtool.OffFlag, ethtool.OffFlag)
		if st.Err != nil {
			return
		}
	}
	st.Status = plugin.StatusOk
	return
}

func (p *pl) checkPauseFeature() (st plugin.State) {
	st = plugin.State{Name: "PauseParam", Status: plugin.StatusFail, Err: nil}
	for _, iface := range p.ifaces {
		pp, err := ethtool.GetPauseParam(iface.name)
		if err != nil {
			st.Err = err
			return
		}
		if pp.RxPause != ethtool.OffFlag {
			st.Status = plugin.StatusDiff
			st.Err = fmt.Errorf("RxPause on %s is %d", iface.name, pp.RxPause)
			return
		}
		if pp.TxPause != ethtool.OffFlag {
			st.Status = plugin.StatusDiff
			st.Err = fmt.Errorf("TxPause on %s is %d", iface.name, pp.TxPause)
			return
		}
	}
	st.Status = plugin.StatusOk
	return
}

func (p *pl) checkLossyRoceFeature() (st plugin.State) {
	st = plugin.State{Name: "LossyRoce", Status: plugin.StatusFail, Err: nil}
	for _, iface := range p.ifaces {
		data, err := mlx.RegGet(iface.pci, lossyRoceRegName)
		if err != nil {
			st.Err = err
			return
		}
		for _, v := range lossyRoceRegFields {
			if (*data)[v] != 1 {
				st.Status = plugin.StatusDiff
				st.Err = fmt.Errorf("%s is %d on %s", v, (*data)[v], iface.pci)
				return
			}
		}
	}
	st.Status = plugin.StatusOk
	return
}

func (p *pl) enableLossyRoceFeature() (st plugin.State) {
	st = plugin.State{Name: "LossyRoce", Status: plugin.StatusFail, Err: nil}
	for _, iface := range p.ifaces {
		data := mlx.RegData{}
		for _, v := range lossyRoceRegFields {
			data[v] = 1
		}
		_, st.Err = mlx.RegSet(iface.pci, lossyRoceRegName, &data)
		if st.Err != nil {
			return
		}
	}
	st.Status = plugin.StatusOk
	return
}
