package sched

import (
	"fmt"

	"a.yandex-team.ru/infra/rsm/sysconf/internal/plugin"
	"a.yandex-team.ru/infra/rsm/sysconf/pkg/fs"
)

const (
	FindIdleOn  = "FIND_IDLE_CPU"
	FindIdleOff = "NO_FIND_IDLE_CPU"
)

var (
	SchedPath = fs.NewDefaultSys().FsPath("kernel/debug/sched_features")
)

func init() {
	plugin.Register(&pl{})
}

type pl struct{}

func (p *pl) Name() string {
	return "sched"
}

func (p *pl) Doc() string {
	return "https://st.yandex-team.ru/RTCNETWORK-740"
}

func (p *pl) IsApplicable() (st plugin.State) {
	st.Status = plugin.StatusEnable
	ok, err := SchedPath.IsFileContains(FindIdleOn)
	if err != nil {
		st.Status = plugin.StatusFail
		st.Err = err
		return
	}
	if !ok {
		st.Status = plugin.StatusSkip
	}
	return
}

func (p *pl) Check() (sts plugin.States) {
	sts.Add(p.checkFindIdleFeature())
	return
}

func (p *pl) Enable(force bool) (sts plugin.States) {
	var st plugin.State
	if st = p.checkFindIdleFeature(); st.Status == plugin.StatusDiff {
		st = p.enableFindIdleFeature()
	}
	sts.Add(st)
	return
}

func (p *pl) Disable(force bool) (sts plugin.States) {
	sts.Add(p.disableFindIdleFeature())
	return
}

func (p *pl) enableFindIdleFeature() (st plugin.State) {
	st = plugin.State{Name: "FindIdleOff", Status: plugin.StatusFail, Err: nil}
	st.Err = SchedPath.WriteFile([]byte(FindIdleOff))
	if st.Err != nil {
		return
	}
	st.Status = plugin.StatusOk
	return
}

func (p *pl) disableFindIdleFeature() (st plugin.State) {
	st = plugin.State{Name: "FindIdleOff", Status: plugin.StatusFail, Err: nil}
	st.Err = SchedPath.WriteFile([]byte(FindIdleOn))
	if st.Err != nil {
		return
	}
	st.Status = plugin.StatusOk
	return
}

func (p *pl) checkFindIdleFeature() (st plugin.State) {
	st = plugin.State{Name: "FindIdleOff", Status: plugin.StatusFail, Err: nil}
	ok, err := SchedPath.IsFileContains(FindIdleOff)
	if err != nil {
		st.Err = err
		return
	}
	if !ok {
		st.Err = fmt.Errorf("%s must content %s", SchedPath.Path(), FindIdleOff)
		st.Status = plugin.StatusDiff
		return
	}
	st.Status = plugin.StatusOk
	return
}
