package scterc

import (
	"fmt"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/infra/rsm/sysconf/internal"
	"a.yandex-team.ru/infra/rsm/sysconf/internal/plugin"
	"a.yandex-team.ru/infra/rsm/sysconf/pkg/smartctl"
)

const (
	MODEL       = "TOSHIBA MG07ACA1"
	SCTERCRTIME = "0"
	SCTERCWTIME = "0"
)

var ()

type pl struct {
	devs []string
}

func init() {
	p := &pl{}
	files, err := filepath.Glob("/sys/bus/scsi/devices/*/block/*/device/model")
	if err != nil {
		panic(err)
	}
	for _, f := range files {
		if err := internal.IsFileContent(f, MODEL); err == nil {
			path := strings.Split(f, "/")
			name := path[len(path)-3]
			p.devs = append(p.devs, name)
		}
	}

	plugin.Register(p)
}

func (p *pl) Name() string {
	return "scterc"
}

func (p *pl) Doc() string {
	return "https://st.yandex-team.ru/RTCNETWORK-585"
}

func (p *pl) IsApplicable() (st plugin.State) {
	st.Status = plugin.StatusEnable
	if len(p.devs) == 0 {
		st.Status = plugin.StatusSkip
	}
	return
}

func (p *pl) Check() (sts plugin.States) {
	sts.Add(p.checkSctErcFeature())
	return
}

func (p *pl) Enable(force bool) (sts plugin.States) {
	var st plugin.State
	if st = p.checkSctErcFeature(); st.Status == plugin.StatusDiff {
		st = p.enableSctErcFeature()
	}
	sts.Add(st)
	return
}

func (p *pl) Disable(force bool) (sts plugin.States) {
	sts.Add(p.disableSctErcFeature())
	return
}

func (p *pl) disableSctErcFeature() (st plugin.State) {
	st = plugin.State{Name: "self", Status: plugin.StatusFail, Err: nil}
	for _, d := range p.devs {
		_, st.Err = smartctl.SctErcSet(d, "0", "0")
		if st.Err != nil {
			return
		}
	}
	st.Status = plugin.StatusOk
	return
}

func (p *pl) enableSctErcFeature() (st plugin.State) {
	st = plugin.State{Name: "self", Status: plugin.StatusFail, Err: nil}
	for _, d := range p.devs {
		_, st.Err = smartctl.SctErcSet(d, SCTERCRTIME, SCTERCWTIME)
		if st.Err != nil {
			return
		}
	}
	st.Status = plugin.StatusOk
	return
}

func (p *pl) checkSctErcFeature() (st plugin.State) {
	st = plugin.State{Name: "self", Status: plugin.StatusFail, Err: nil}
	for _, d := range p.devs {
		data, err := smartctl.SctErcGet(d)
		if err != nil {
			st.Err = err
			return
		}
		if data["Read"] != SCTERCRTIME {
			st.Status = plugin.StatusDiff
			st.Err = fmt.Errorf("%s smart scterc ReadTime %s, must %s", d, data["Read"], SCTERCRTIME)
			return
		}
		if data["Write"] != SCTERCWTIME {
			st.Status = plugin.StatusDiff
			st.Err = fmt.Errorf("%s smart scterc WriteTime %s, must %s", d, data["Write"], SCTERCWTIME)
			return
		}
	}
	st.Status = plugin.StatusOk
	return
}
