package sigbus

import (
	"io/ioutil"
	"path/filepath"

	"a.yandex-team.ru/infra/rsm/sysconf/internal"
	"a.yandex-team.ru/infra/rsm/sysconf/internal/plugin"
)

const (
	boardFile           = "/sys/devices/virtual/dmi/id/board_name"
	boardFileContent    = "MY81-EX0-Y3N"
	tolerantFileRgxp    = "/sys/devices/system/machinecheck/machinecheck*/tolerant"
	tolerantFileContent = "0"
)

type pl struct{}

func init() {
	plugin.Register(&pl{})
}

func (p *pl) Name() string {
	return "sigbus"
}

func (p *pl) Doc() string {
	return "https://st.yandex-team.ru/RTCSUPPORT-6085"
}

func (p *pl) IsApplicable() (st plugin.State) {
	st.Status = plugin.StatusEnable
	if st.Err = internal.IsFileContent(boardFile, boardFileContent); st.Err != nil {
		st.Status = plugin.StatusSkip
	}
	return
}

func (p *pl) Check() (sts plugin.States) {
	sts.Add(checkTolerantFeature())
	return
}

func (p *pl) Enable(force bool) (sts plugin.States) {
	var st plugin.State
	if st = checkTolerantFeature(); st.Status == plugin.StatusDiff {
		st = enableTolerantFeature()
	}
	sts.Add(st)
	return
}

func (p *pl) Disable(force bool) (sts plugin.States) {
	sts.Add(plugin.State{Name: "self", Status: plugin.StatusSkip, Err: plugin.ErrNotImpl})
	return
}

func enableTolerantFeature() (st plugin.State) {
	st = plugin.State{Name: "Tolerant", Status: plugin.StatusFail, Err: nil}
	files, err := filepath.Glob(tolerantFileRgxp)
	if err != nil {
		st.Err = err
		return
	}
	for _, f := range files {
		st.Err = ioutil.WriteFile(f, []byte(tolerantFileContent), 0)
		if st.Err != nil {
			return
		}
	}
	st.Status = plugin.StatusOk
	return
}

func checkTolerantFeature() (st plugin.State) {
	st = plugin.State{Name: "Tolerant", Status: plugin.StatusFail, Err: nil}
	files, err := filepath.Glob(tolerantFileRgxp)
	if err != nil {
		st.Err = err
		return
	}
	for _, f := range files {
		st.Err = internal.IsFileContent(f, tolerantFileContent)
		if st.Err != nil {
			st.Status = plugin.StatusDiff
			return
		}
	}
	st.Status = plugin.StatusOk
	return
}
