package xpsoff

import (
	"fmt"
	"regexp"

	"a.yandex-team.ru/infra/rsm/sysconf/internal"
	"a.yandex-team.ru/infra/rsm/sysconf/internal/plugin"
	"a.yandex-team.ru/infra/rsm/sysconf/pkg/fs"
)

func init() {
	p, err := New()
	if err != nil {
		panic(err)
	}
	plugin.Register(p)
}

const (
	xpsCpusName = "xps_cpus"
)

var (
	rgxp = regexp.MustCompile("^(enx|eth[0-9])")
)

type pl struct {
	netDevs fs.NetDevList
}

func New() (*pl, error) {
	// net
	netDevs, err := fs.NewDefaultSys().ClassNet()
	if err != nil {
		return nil, err
	}

	netDevs = netDevs.FilterName(rgxp).FilterOperstateUp(true)

	return &pl{
		netDevs: netDevs,
	}, nil
}

func (p *pl) Name() string {
	return "xpsoff"
}

func (p *pl) Doc() string {
	return "https://st.yandex-team.ru/RTCNETWORK-859"
}

func (p *pl) IsApplicable() (st plugin.State) {
	st.Status = plugin.StatusEnable
	return
}

func (p *pl) Check() (sts plugin.States) {
	for _, netDev := range p.netDevs {
		sts.Add(p.checkXPSOffFeature(netDev))
	}
	return
}

func (p *pl) Enable(force bool) (sts plugin.States) {
	var st plugin.State
	for _, netDev := range p.netDevs {
		if st = p.checkXPSOffFeature(netDev); st.Status == plugin.StatusDiff {
			st = p.enableXPSOffFeature(netDev)
		}
		sts.Add(st)
	}
	return
}

func (p *pl) Disable(force bool) (sts plugin.States) {
	sts.Add(plugin.State{Name: "self", Status: plugin.StatusSkip, Err: plugin.ErrNotImpl})
	return
}

// TODO
func (p *pl) checkXPSOffFeature(netDev *fs.NetDev) (st plugin.State) {
	model, _ := netDev.Model()
	if model == "0x10d3" {
		st.Status = plugin.StatusSkip
		st.Err = plugin.ErrNotImpl
		return
	}
	st = plugin.State{Name: fmt.Sprintf("%s.%s", netDev.Name, "XPSOff"), Status: plugin.StatusFail, Err: nil}
	for _, txQueue := range netDev.Queues.Tx {
		xpsCpusPath := txQueue.Path(xpsCpusName)
		cbmCurrent, err := internal.GetCPUBitMapFromFile(xpsCpusPath)
		if err != nil {
			st.Err = err
			return
		}
		if cbmCurrent != "" {
			st.Status = plugin.StatusDiff
			st.Err = fmt.Errorf("%s is not disabled", xpsCpusPath)
			return
		}
	}
	st.Status = plugin.StatusOk
	return
}

// TODO
func (p *pl) enableXPSOffFeature(netDev *fs.NetDev) (st plugin.State) {
	st = plugin.State{Name: fmt.Sprintf("%s.%s", netDev.Name, "XPSOff"), Status: plugin.StatusFail, Err: nil}
	for _, txQueue := range netDev.Queues.Tx {
		xpsCpusFile := txQueue.FsPath(xpsCpusName)
		st.Err = xpsCpusFile.WriteFile([]byte("0"))
		if st.Err != nil {
			return
		}
	}
	st.Status = plugin.StatusOk
	return
}
