package internal

import (
	"fmt"
	"io/ioutil"
	"strconv"
	"strings"
)

func MinUInt(x, y uint32) uint32 {
	if x > y {
		return y
	}
	return x
}

// Generate map like this f0,00ff00ff, where
// 96-128   64-95    32-63     0-31
// 00000000,00000000,000000f0,00ff00ff
// CPU    BIN     HEX
// 0   0000 0001   1
// 1   0000 0010   2
// 2   0000 0100   4
// 3   0000 1000   8
// 4   0001 0000  10
// 5   0010 0000  20
// 6   0100 0000  40
// 7   1000 0000  80
func CpusToCPUBitMap(cpuCount uint32, cpus []uint32) string {
	var cbmS strings.Builder
	cbm := make([]uint32, cpuCount/32+1)
	for _, cpu := range cpus {
		cbm[cpu/32] |= 1 << (cpu % 32)
	}
	for i := len(cbm) - 1; i >= 0; i-- {
		fmt.Fprintf(&cbmS, ",%08x", cbm[i])
	}
	return strings.TrimLeft(cbmS.String(), "0,")
}

func CPUBitmapToCpus(s string) (cpus []uint32, err error) {
	cbmList := strings.Split(s, ",")
	for i := range cbmList {
		// reverse mode
		bmStr := cbmList[len(cbmList)-1-i]
		bm, err := strconv.ParseInt(bmStr, 16, 64)
		if err != nil {
			return nil, err
		}
		for cur := uint(0); cur < 32; cur++ {
			if int32(bm)&(1<<cur) != 0 {
				cpus = append(cpus, uint32(cur+32*uint(i)))
			}
		}
	}
	return
}

func GetCPUBitMapFromFile(pth string) (string, error) {
	data, err := ioutil.ReadFile(pth)
	if err != nil {
		return "", err
	}
	return strings.TrimLeft(strings.Trim(string(data[:]), "\n "), "0,"), nil
}
