package ethtool

import (
	"golang.org/x/sys/unix"
	"unsafe"
)

const (
	IfaceNameSize         = 16
	SiocEthTool           = 0x8946
	OffFlag               = 0x0
	OnFlag                = 0x1
	EthToolGPauseParam    = 0x00000012
	EthToolSPauseParam    = 0x00000013
	EthToolGRingParam     = 0x00000010
	EthToolSRingParam     = 0x00000011
	EthToolGChannelsParam = 0x0000003c
	EthToolSChannelsParam = 0x0000003d
)

type ifreq struct {
	name [IfaceNameSize]byte
	data uintptr
}

type PauseParam struct {
	cmd     uint32
	Autoneg uint32
	RxPause uint32
	TxPause uint32
}

type RingParam struct {
	cmd        uint32
	RxMax      uint32
	RxMiniMax  uint32
	RxJumboMax uint32
	TxMax      uint32
	Rx         uint32
	RxMini     uint32
	RxJumbo    uint32
	Tx         uint32
}

type ChannelsParam struct {
	cmd         uint32
	RxMax       uint32
	TxMax       uint32
	OtherMax    uint32
	CombinedMax uint32
	Rx          uint32
	Tx          uint32
	Other       uint32
	Combined    uint32
}

func GetPauseParam(iface string) (*PauseParam, error) {
	p := &PauseParam{
		cmd: EthToolGPauseParam,
	}

	if err := ioctl(iface, uintptr(unsafe.Pointer(p))); err != nil {
		return nil, err
	}

	return p, nil
}

func SetPauseParam(iface string, rx uint32, tx uint32) (*PauseParam, error) {
	p := &PauseParam{
		cmd:     EthToolSPauseParam,
		RxPause: rx,
		TxPause: tx,
	}

	if err := ioctl(iface, uintptr(unsafe.Pointer(p))); err != nil {
		return nil, err
	}

	return p, nil
}

func GetRingParam(iface string) (*RingParam, error) {
	p := &RingParam{
		cmd: EthToolGRingParam,
	}

	if err := ioctl(iface, uintptr(unsafe.Pointer(p))); err != nil {
		return nil, err
	}

	return p, nil
}

func SetRingParam(iface string, rx uint32, mini uint32, jumbo uint32, tx uint32) (*RingParam, error) {
	p := &RingParam{
		cmd:     EthToolSRingParam,
		Rx:      rx,
		RxMini:  mini,
		RxJumbo: jumbo,
		Tx:      tx,
	}

	if err := ioctl(iface, uintptr(unsafe.Pointer(p))); err != nil {
		return nil, err
	}

	return p, nil
}

func GetChannelsParam(iface string) (*ChannelsParam, error) {
	p := &ChannelsParam{
		cmd: EthToolGChannelsParam,
	}

	if err := ioctl(iface, uintptr(unsafe.Pointer(p))); err != nil {
		return nil, err
	}

	return p, nil
}

func SetChannelsParam(iface string, rx uint32, tx uint32, other uint32, comb uint32) (*ChannelsParam, error) {
	p := &ChannelsParam{
		cmd:      EthToolSChannelsParam,
		Rx:       rx,
		Tx:       tx,
		Other:    other,
		Combined: comb,
	}

	if err := ioctl(iface, uintptr(unsafe.Pointer(p))); err != nil {
		return nil, err
	}

	return p, nil
}

func ioctl(iface string, data uintptr) error {
	fd, err := unix.Socket(unix.AF_INET, unix.SOCK_DGRAM, unix.IPPROTO_IP)
	if err != nil {
		return err
	}
	fd = int(fd)
	defer unix.Close(fd)

	var name [IfaceNameSize]byte
	copy(name[:], []byte(iface))

	ifr := ifreq{
		name: name,
		data: data,
	}

	_, _, ep := unix.Syscall(unix.SYS_IOCTL, uintptr(fd), SiocEthTool, uintptr(unsafe.Pointer(&ifr)))
	if ep != 0 {
		return ep
	}

	return nil
}
