package fs

import (
	"bytes"
	"io/ioutil"
	"os"
	"path/filepath"
)

type Fs string

// implement Stringer interface
func (fs Fs) String() string {
	return string(fs)
}

func (fs Fs) Path(p ...string) string {
	return filepath.Join(append([]string{fs.String()}, p...)...)
}

func (fs Fs) FsPath(p ...string) Fs {
	return Fs(fs.Path(p...))
}

func (fs Fs) Base(p ...string) string {
	return filepath.Base(fs.Path(p...))
}

func (fs Fs) Glob(p ...string) ([]string, error) {
	return filepath.Glob(fs.Path(p...))
}

func (fs Fs) ReadDir(p ...string) ([]os.FileInfo, error) {
	return ioutil.ReadDir(fs.Path(p...))
}

func (fs Fs) ReadFile(p ...string) ([]byte, error) {
	return ioutil.ReadFile(fs.Path(p...))
}

func (fs Fs) WriteFile(data []byte, p ...string) error {
	return ioutil.WriteFile(fs.Path(p...), data, 0)
}

func (fs Fs) IsFileEqual(ptrn string, p ...string) (bool, error) {
	data, err := fs.ReadFile(p...)
	if err != nil {
		return false, err
	}
	return bytes.Equal(bytes.TrimSpace(data[:]), []byte(ptrn)), nil
}

func (fs Fs) IsFileContains(ptrn string, p ...string) (bool, error) {
	data, err := fs.ReadFile(p...)
	if err != nil {
		return false, err
	}
	return bytes.Contains(data[:], []byte(ptrn)), nil
}

func ParseFs(paths []string) []Fs {
	fslist := []Fs{}
	for _, path := range paths {
		fslist = append(fslist, Fs(path))
	}
	return fslist
}
