package fs

import (
	"bufio"
	"bytes"
	"io"
	"os"
	"regexp"
	"strconv"
	"strings"

	"a.yandex-team.ru/infra/rsm/sysconf/pkg/setpci"
)

const (
	defaultSysPath      = "/sys"
	classInfinibandPath = "class/infiniband"
	classNetPath        = "class/net"
	// attribute for tunning MaxReadRequest on PCI devices
	attrMaxReadReq = "68.w"
)

type Sys struct {
	Fs
}

func NewDefaultSys() *Sys {
	return NewSys(defaultSysPath)
}

func NewSys(p string) *Sys {
	return &Sys{Fs: Fs(p)}
}

func (fs Sys) ClassInfiniband() (map[string]*IBDev, error) {
	result := map[string]*IBDev{}

	files, err := fs.ReadDir(classInfinibandPath)
	if err != nil && !os.IsNotExist(err) {
		return nil, err
	}
	for _, i := range files {
		dev, err := NewIBDev(fs.FsPath(classInfinibandPath, i.Name()))
		if err != nil {
			return nil, err
		}
		result[i.Name()] = dev
	}
	return result, nil
}

func (fs Sys) ClassNet() (NetDevList, error) {
	result := NetDevList{}

	files, err := fs.ReadDir(classNetPath)
	if err != nil && !os.IsNotExist(err) {
		return nil, err
	}

	for _, i := range files {
		dev, err := NewNetDev(fs.FsPath(classNetPath, i.Name()))
		if err != nil {
			return nil, err
		}
		result = append(result, dev)
	}

	return result, nil
}

type PCIDev struct {
	Driver string
	Slot   string
	Class  string
	Vendor string
	Device string
}

func ReadPCIDev(r io.Reader) *PCIDev {
	dev := &PCIDev{}
	scanner := bufio.NewScanner(r)
	for scanner.Scan() {
		data := strings.Split(strings.TrimSpace(scanner.Text()), "=")
		if len(data) != 2 {
			continue
		}
		switch data[0] {
		case "DRIVER":
			dev.Driver = data[1]
		case "PCI_CLASS":
			dev.Class = data[1]
		case "PCI_ID":
			id := strings.Split(data[1], ":")
			dev.Vendor = id[0]
			dev.Device = id[1]
		case "PCI_SLOT_NAME":
			dev.Slot = data[1]
		}
	}
	return dev
}

func (dev *PCIDev) IsMaxReadReq(v int) (bool, error) {
	data, err := setpci.Get(dev.Slot, attrMaxReadReq)
	if err != nil {
		return false, err
	}
	return bytes.Equal([]byte(strconv.Itoa(v)), data), nil
}

// Setting PCIe MaxReadRequest.
// See more https://community.mellanox.com/s/article/understanding-pcie-configuration-for-maximum-performance
func (dev *PCIDev) SetMaxReadReq(v int) error {
	_, err := setpci.Set(dev.Slot, attrMaxReadReq, strconv.Itoa(v))
	return err
}

type IBDev struct {
	Fs   Fs
	Name string
	PCI  *PCIDev
}

func (dev *IBDev) SetNodeDesc(data []byte) error {
	return dev.Fs.WriteFile(data, "node_desc")
}

func NewIBDev(p Fs) (*IBDev, error) {
	uevent, err := p.ReadFile("device/uevent")
	if err != nil {
		return nil, err
	}

	return &IBDev{
		Fs:   p,
		Name: p.Base(),
		PCI:  ReadPCIDev(bytes.NewReader(uevent)),
	}, nil
}

type NetDevList []*NetDev

func (netDevs NetDevList) FilterOperstateUp(isUp bool) (result NetDevList) {
	for _, netDev := range netDevs {
		if netDev.IsUp == isUp {
			result = append(result, netDev)
		}
	}

	return
}

func (netDevs NetDevList) FilterName(rgxp *regexp.Regexp) (result NetDevList) {
	for _, netDev := range netDevs {
		if rgxp.MatchString(netDev.Name) {
			result = append(result, netDev)
		}
	}

	return
}

type Queues struct {
	Rx []Fs
	Tx []Fs
}

func GetQueues(p Fs) (*Queues, error) {
	rx, err := p.Glob("rx-*")
	if err != nil {
		return nil, err
	}

	tx, err := p.Glob("tx-*")
	if err != nil {
		return nil, err
	}

	return &Queues{
		Rx: ParseFs(rx),
		Tx: ParseFs(tx),
	}, nil
}

type NetDev struct {
	Fs     Fs
	Name   string
	IsUp   bool
	Queues *Queues
}

func (netDev NetDev) Model() (string, error) {
	data, err := netDev.Fs.FsPath("device/device").ReadFile()
	if err != nil {
		return "", nil
	}
	return strings.Trim(string(data), "\n "), nil
}

func NewNetDev(p Fs) (*NetDev, error) {
	isUp, err := p.FsPath("operstate").IsFileContains("up")
	if err != nil {
		return nil, err
	}

	queues, err := GetQueues(p.FsPath("queues"))
	if err != nil {
		return nil, err
	}

	return &NetDev{
		Fs:     p,
		Name:   p.Base(),
		IsUp:   isUp,
		Queues: queues,
	}, nil
}
