#!/usr/bin/env python
# Provides: porto_coredumps
import json

import porto


SERVICE_NAME = 'porto_coredumps'

OK = 'OK'
WARN = 'WARN'
CRIT = 'CRIT'


def _exc_to_crit(reason, e):
    """
    :type e: Exception
    :rtype: (str, str)
    """
    msg = unicode(e)
    return CRIT, u'{}: {}'.format(reason, msg[:300])


def _list_root_container_names(conn):
    """
    :type conn: porto.Connection
    :rtype: list[str]
    """
    rv = []
    for container_name in conn.List():
        # This is how portoctl finds toplevel containers:
        # https://bb.yandex-team.ru/projects/PORTO/repos/porto/browse/src/portoctl.cpp#1549
        if '/' not in container_name:
            rv.append(container_name)
    rv.sort()
    return rv


def check():
    # Useful links:
    # https://bb.yandex-team.ru/projects/PORTO/repos/porto/browse/test/test-coredump.py
    # https://wiki.yandex-team.ru/porto/coredump/

    try:
        conn = porto.Connection()
        conn.Connect()
    except Exception as e:
        return _exc_to_crit(u'Failed to connect to porto', e)

    try:
        root_container_names = _list_root_container_names(conn)
    except Exception as e:
        return _exc_to_crit(u'Failed to list root porto containers', e)

    for root_container_name in root_container_names:
        try:
            try:
                cores_total_raw = conn.GetProperty(root_container_name, 'CORE.total')
            except porto.exceptions.LabelNotFound:
                cores_total = 0
            else:
                cores_total = int(cores_total_raw)
        except Exception as e:
            return _exc_to_crit(u'Failed to get property "CORE.total" of '
                                u'root container "{}"'.format(root_container_name), e)

        try:
            try:
                cores_dumped_raw = conn.GetProperty(root_container_name, 'CORE.dumped')
            except porto.exceptions.LabelNotFound:
                cores_dumped = 0
            else:
                cores_dumped = int(cores_dumped_raw)
        except Exception as e:
            return _exc_to_crit(u'Failed to get property "CORE.dumped" of '
                                u'root container "{}"'.format(root_container_name), e)

        if cores_total > 0:
            return CRIT, u'Cores found (total: {}, dumped: {})'.format(cores_total, cores_dumped)
        else:
            return OK, u''


def main():
    status, desc = check()
    print json.dumps({
        'events': [
            {
                'service': SERVICE_NAME,
                'description': desc,
                'status': status,
            },
        ],
    })


if __name__ == '__main__':
    main()
