#!/usr/bin/env python
# encoding: utf-8

from datetime import datetime
import dateutil.parser
import requests
import json
import re
import os
from pprint import pprint
from os.path import realpath, join
import argparse
from time import sleep

import nirvana.job_context as nv
from startrek_client import Startrek


CERT_PATH = realpath(join(__file__, '../allCAs.pem'))  # Скачать можно здесь "https://crls.yandex.net/allCAs.pem"


class CredentialNotFound(Exception):
    pass


action_items_main = re.compile("(=+ *Меры предотвращения.*?==.*?=|=+ *Методы предотвращения.*?==.*?=|=+ *Action."
                               "*?==.*?=)", re.DOTALL | re.IGNORECASE)
action_items_fallback = re.compile("(=+ *Меры предотвращения.*|=+ *Методы предотвращения.*|=+ *Action.*)",
                                   re.DOTALL | re.IGNORECASE)
ticket_regex = re.compile("[A-Z]+-[0-9]+", re.DOTALL)
treker_date_format = "%Y-%m-%dT%H:%M:%S.%f+0000"


def choose_credential(value, envvar, path):
    """
    Выбирает аутентификационные данные, в порядке приоритета:
         либо явно заданные,
         либо из переменной окружения,
         либо из заданного файла
    :param str value: Заданное пользователем значение
    :param str envvar: Переменная окружения, в которую может быть записано значение
    :param str path: Путь, в котором может лежать значение
    :rtype: str
    :return: Выбранное значение
    """
    if value is None:
        value = os.getenv(envvar)  # В нирвана операциях используется так

    path = os.path.expanduser(path)
    if value is None and os.path.exists(path):
        value = open(path).read().strip()  # Можно использовать для локальных тестов

    if value is None:
        msg = 'credential not found in "{}" nor "{}"'.format(envvar, path)
        raise CredentialNotFound(msg)
    else:
        return value


def make_oauth_header(token, envvar, path, content_type="application/json; charset=utf-8"):
    # Словарь хэдеров, для аутентификации через http oauth
    token = choose_credential(token, envvar, path)
    return {"Content-Type": content_type,
            "Authorization": "OAuth {}".format(token)}


def find_users(dep_name, token):
    users = set()
    gr_adr = "https://staff-api.yandex-team.ru/v3/persons?department_group.url=" + dep_name
    gr_req = requests.get(gr_adr, headers={'Authorization': "OAuth " + token})
    gr_staff = gr_req.json()
    users |= set([person['login'] for person in gr_staff['result']])

    sub_adr = "https://staff-api.yandex-team.ru/v3/groups?parent.url=" + dep_name
    sub_req = requests.get(sub_adr, headers={'Authorization': "OAuth " + token})
    sub_staff = sub_req.json()
    for dep in sub_staff['result']:
        users |= find_users(dep['url'], token)
    return users


def find_rtc_services():
    adr = "http://workplace.z.yandex-team.ru/api/workplace.services.CatalogService/listRecords?vertical=rtc"
    req = requests.post(adr)
    staff = req.json()
    return [obj['service'] for obj in staff["objects"]]


def short_data(dt):
    return dateutil.parser.parse(dt).replace(tzinfo=None).strftime("%Y-%m-%d")


def find_service(lis):
    return [ser.lower()[8:] for ser in issue.tags if ser.lower().startswith('service:')]


def make_queue(dom_list, start_date=None, end_date=None):
    ans = "("
    for dom in dom_list[:-1]:
        ans += "Queue: " + dom + " OR "
    ans += "Queue: " + dom_list[-1] + ")"
    if start_date:
        ans += " AND Created:>=" + start_date
    if end_date:
        ans += " AND Created:<=" + end_date
    return ans


def str2bool(v):
    if isinstance(v, bool):
        return v
    if v.lower() in ('yes', 'true', 't', 'y', '1'):
        return True
    elif v.lower() in ('no', 'false', 'f', 'n', '0'):
        return False
    else:
        raise argparse.ArgumentTypeError('Boolean value expected.')


def extract_action_items_number(descr):

    try:
        action_items_text = re.search(action_items_main, descr).group(0)
    except AttributeError:
        try:
            action_items_text = re.search(action_items_fallback, descr).group(0)
        except AttributeError:
            return 0
    except:
        return 0

    return len(set([i.group(0) for i in list(re.finditer(ticket_regex, action_items_text))]))


def check_if_issue_spproblem(issue):
    links = issue.links.get_all()

    for lnk in links:
        ticket_id = str(lnk.object.key)
        direction = str(lnk.direction)

        if direction == "outward":
            if "spproblem" in ticket_id.lower():
                return True

    return False


def get_verical_mapping():
    workplace_base = "https://workplace.z.yandex-team.ru/api/workplace.services.CatalogService/listRecords"
    payload = {}
    vertical_mapping = {}

    result = requests.post(workplace_base, json=payload).json()["objects"]

    for el in result:
        service = el.get("service")

        if service:
            vertical = el.get("vertical", "")
            vertical_mapping[service] = vertical

    return vertical_mapping


def calculate_spi_lifetime(issue):

    created, resolved = issue.createdAt, issue.resolvedAt

    if not resolved:
        return None
    else:
        return (datetime.strptime(resolved, treker_date_format) - datetime.strptime(created, treker_date_format)).days


def get_mode(mode_env_name):

    mode_var = os.getenv(mode_env_name)

    if mode_var:
        return mode_var
    else:
        return "DEFAULT"


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    rtc_list, service_vertical_mapping = find_rtc_services(), get_verical_mapping()
    parser.add_argument('--token', help='TRACKER TOKEN', type=str, default=None)
    parser.add_argument('-sd', '--start_date', help='QUEUES STARTDATE, DEF=2019-01-01', type=str, default='2019-01-01')
    parser.add_argument('-ed', '--end_date', help='QUEUES ENDDATE, DEF=today()', type=str, default='today()')
    parser.add_argument('-q', '--queues', help='QUEUES, DEF=SPI', nargs='+', default=['SPI'])
    parser.add_argument('--gna', help='INTERSECT WITH SERVICES, DEF=LONGLIST', nargs='+', default=rtc_list)
    parser.add_argument("--need_gna", type=str2bool, default=True, help="ACTIVATE INTERSECT WITH SERVICES, DEF=true")
    parser.add_argument('-dep', '--department',
                        help='INTERSECT WITH DEPARTMENT FOLLOWERS, DEF=yandex_mnt_sa_runtime_mondev',
                        type=str, default='yandex_mnt_sa_runtime_mondev')
    parser.add_argument("--need_fol", type=str2bool, default=False,
                        help="ACTIVATE INTERSECT WITH DEPARTMENT FOLLOWERS, DEF=false")
    parser.add_argument("--data_in_isa", type=str2bool, default=False,
                        help="ACTIVATE INTERSECT WITH DEPARTMENT FOLLOWERS, DEF=true")
    parser.add_argument("--local_run", dest="local_run", action="store_true")
    parser.add_argument("--nirvana_run", dest="local_run", action="store_false")
    parser.add_argument("--test_run", dest="test_run", action="store_true")
    parser.add_argument("--full_run", dest="test_run", action="store_false")
    parser.set_defaults(local_run=False)
    parser.set_defaults(test_run=False)
    args = parser.parse_args()
    token_get = args.token
    our_gna = set(args.gna)
    local_run, test_run = args.local_run, args.test_run

    token = choose_credential(token_get, 'TRACKER_TOKEN', '~/.st/token')
    user = "username"
    url2 = 'https://st-api.yandex-team.ru'
    data_file_name = "data_file.json"
    mode = get_mode("MODE")

    for attempt in range(100):
        try:
            client = Startrek(useragent=user, base_url=url2, token=token)

            all_dom = args.queues

            set_users = find_users(args.department, token)

            queue = make_queue(all_dom, args.start_date, args.end_date)
            issues = client.issues.find(queue)

            data_in_ISA = args.data_in_isa
            ans_list, issues_list, counter = [], [], 0

            for issue in issues:
                set_follower = set([fol.login for fol in issue.followers])
                assignee = issue.assignee.login if issue.assignee else "None"
                service_names = find_service(issue.tags)

                try:
                    resolved = short_data(issue.resolvedAt)
                except:
                    resolved = None

                spi_added = False

                for service_name in service_names:
                    if not (assignee in set_users or len(set_follower & set_users) > 0) and service_name in our_gna:
                        pass
                    if (not args.need_gna or service_name in our_gna) and \
                            (not args.need_fol or (assignee in set_users or len(set_follower & set_users) > 0)):
                        ans_list.append({"key": issue.key,
                                         "fielddate": issue.createdAt if data_in_ISA else short_data(issue.createdAt),
                                         "status": issue.status.key, "service": service_name,
                                         "vertical": service_vertical_mapping[service_name]})
                        if spi_added is False:
                            issues_list.append(
                                {"key": issue.key, "has_spproblems_linked": check_if_issue_spproblem(issue),
                                 "lifetime_days": calculate_spi_lifetime(issue),
                                 "action_items": extract_action_items_number(issue.description),
                                 "created_at": short_data(issue.createdAt),
                                 "resolved_at": resolved})
                            spi_added = True

                counter += 1

                if test_run is True:
                    if counter == 30:
                        break

            if local_run is False:
                job_context = nv.context()
                outputs = job_context.get_outputs()

                with open(outputs.get(data_file_name), "w") as write_file:
                    if mode == "DEFAULT":
                        json.dump(ans_list, write_file)
                    elif mode == "ISSUE":
                        json.dump(issues_list, write_file)

            else:
                with open(data_file_name, "w") as write_file:
                    if mode == "DEFAULT":
                        json.dump(ans_list, write_file)
                    elif mode == "ISSUE":
                        json.dump(issues_list, write_file)
            break
        except:
            print("unsuccessful attempt")
            sleep(attempt)
            continue
