import datetime
import argparse
from collections import Counter
import requests
import os
import yaml
import library.python.resource as rs
import json
import nirvana.job_context as nv


from startrek_client import Startrek

envstats_list = ["ACTIVATING",
                "ALLOCATING",
                "ALLOCATED",
                "COMMITTED",
                "CONFIGURING",
                "DEPLOYED",
                "DEPLOYING",
                "FAILED",
                "NEW",
                "PREPARED",
                "PREPARING",
                "REMOVED",
                "REMOVING",
                "SEMI_ACTIVATED",
                "SEMI_PREPARED",
                "TEMPLATE"]


def get_abc_heads(abc_id):

    _headers = {'Authorization': 'OAuth {}'.format(os.environ.get("ABC_TOKEN"))}
    url = "https://abc-back.yandex-team.ru/api/v4/services/responsibles/?service={0}".format(str(abc_id))

    result = requests.get(url, headers=_headers).json()

    if result.get("results"):
        return result["results"][0]["person"]["login"]

    return None


def str2bool(v):
    if isinstance(v, bool):
        return v
    if v.lower() in ('yes', 'true', 't', 'y', '1'):
        return True
    elif v.lower() in ('no', 'false', 'f', 'n', '0'):
        return False
    else:
        raise argparse.ArgumentTypeError('Boolean value expected.')


def get_abc_slug(abc_id):
    _headers = {'Authorization': 'OAuth {}'.format(os.environ.get("ABC_TOKEN"))}
    url = "https://abc-back.yandex-team.ru/api/v4/services/{0}/?fields=name".format(abc_id)

    result = requests.get(url, headers=_headers).json()

    if result["name"].get("ru"):
        return result["name"]["ru"]
    else:
        return result["name"]["en"]


def get_ticket_map():

    mapping = {}

    issues = Startrek(useragent="akorobkov", token=os.environ.get("STARTREK_TOKEN")).issues.find(filter={
        "queue": "DECOMMISSIONING", "tags": "qloud-int-decommissioning", "resolution": "empty()"
    })

    issues.extend(Startrek(useragent="akorobkov", token=os.environ.get("STARTREK_TOKEN")).issues.find(filter={
        "queue": "DECOMMISSIONING", "tags": "qloud-ext-decommissioning", "resolution": "empty()"
    }))

    for ticket in issues:
        mapping[ticket.key] = ticket.assignee.id if ticket.assignee else None

    return mapping


def calculate_qloud_enviroments():

    qloud_enviroments_mapping = {
        k: {
            "ext": Counter(),
            "int": Counter()
        } for k in envstats_list
    }

    qloud_total_mapping = {
        "ext": Counter(),
        "int": Counter()
    }

    _headers = {"Authorization": f"OAuth {os.environ.get('QLOUD_TOKEN')}"}

    base_urls = {
        "ext": "https://qloud-ext.yandex-team.ru/api/v1/project/",
        "int": "https://qloud.yandex-team.ru/api/v1/project/"
    }

    for profile, url in base_urls.items():
        result = requests.get(url, headers=_headers).json()

        for record in result:
            env_by_stat = {k: 0 for k in envstats_list}
            envtotal = 0

            try:
                enviroments = record["environmentStatuses"]
                abc = str(record["metaInfo"]["abcId"])
            except KeyError:
                continue

            for envname, envcount in enviroments.items():
                env_by_stat[envname] += envcount
                envtotal += envcount

            for envname in qloud_enviroments_mapping:
                if env_by_stat[envname] > 0:
                    qloud_enviroments_mapping[envname][profile].update({abc: env_by_stat[envname]})

            if envtotal > 0:
                qloud_total_mapping[profile].update({abc: envtotal})

    return qloud_enviroments_mapping, qloud_total_mapping


def extract_ticket_abc_mapping():
    return {v: k for k, v in yaml.load(rs.find("/abc_ticket_relation.yaml").decode("utf-8")).items()}


def main(without_profile):
    result = []

    ticket_abc_mapping = extract_ticket_abc_mapping()
    ticket_assignee_mapping = get_ticket_map()
    abc_enviroment_mapping, abc_total_mapping = calculate_qloud_enviroments()

    for ticket, assignee in ticket_assignee_mapping.items():
        if not assignee:
            abc_responsible = get_abc_heads(ticket_abc_mapping[ticket])

            ticket_assignee_mapping[ticket] = abc_responsible if abc_responsible else "dmitriyt"

    for envname, submap in abc_enviroment_mapping.items():

        for profile, abc_counter in submap.items():

            for ticket, abc in ticket_abc_mapping.items():

                if str(abc) not in abc_counter:
                    continue

                if ticket not in ticket_assignee_mapping:
                    continue

                enviroments = abc_counter[str(abc)]
                envtot = abc_total_mapping[profile][str(abc)]

                GY_BORDER = 10
                YR_BORDER = 50

                if envtot < GY_BORDER:
                    status = "green"
                elif envtot >= GY_BORDER and envtot < YR_BORDER:
                    status = "yellow"
                else:
                    status = "red"

                slug = get_abc_slug(abc)

                record = {
                    "assignee": ticket_assignee_mapping[ticket],
                    "link": "https://st.yandex-team.ru/" + ticket,
                    "product": slug,
                    "status": status,
                    "status_str": "{} Qloud enviroments are still open".format(str(enviroments)),
                    "envname": envname,
                    "tasks_count": enviroments,
                    "profile": profile
                }

                if os.environ.get("WITH_DATE"):
                    record["date"] = datetime.date.today().strftime("%Y-%m-%d")

                result.append(record)

    if without_profile:
        grouped_data = {}

        for record in result:
            if record["product"] not in grouped_data:
                del record["profile"]
                grouped_data[record["product"]] = record
            else:
                grouped_data[record["product"]]["tasks_count"] += record["tasks_count"]

        return list(grouped_data.values())
    else:
        return result


if __name__ == '__main__':

    parser = argparse.ArgumentParser()
    parser.add_argument("--local_run", type=str2bool, default=False)
    args = parser.parse_args()

    if args.local_run is False:
        job_context = nv.context()
        parameters, outputs = job_context.get_parameters(), job_context.get_outputs()

        with_profile = parameters.get(os.environ.get("WITHOUT_PROFILE"))
        data = main(with_profile)

        with open(outputs.get(os.environ.get("OUTPUT_FILE")), "w") as write_file:
            json.dump(data, write_file)

    else:
        with_profile = None
        data = main(with_profile)

        with open("test_data.json", "w") as fle:
            json.dump(data, fle)
