import json
import time
import argparse

import infra.analytics.io_limits_pipeline.utils as utils
import nirvana.job_context as nv
from infra.rtc.analytics.gpu_metrics_collect.utils import safe_div


def get_last_week_data(rows):
    return [row for row in rows if time.time() - time.mktime(time.strptime(row["time"], "%Y-%m-%d %H:%M:%S")) <= 7 * 24 * 60 * 60]


def aggregate_week_ago(rows):
    timestmp = time.strftime("%H:minutes_here", time.localtime())
    mins = time.strftime("%M", time.localtime())
    timestmp = timestmp.replace("minutes_here", "30" if int(mins) >= 30 else "00")
    if timestmp != "00:00":
        return []

    week_ago = time.strftime("%Y-%m-%d", time.localtime(time.time() - 8 * 24 * 60 * 60))
    res_pre = {}
    for row in rows:
        date = time.strftime("%Y-%m-%d", time.strptime(row["time"], "%Y-%m-%d %H:%M:%S"))
        if date == week_ago:
            walle = row["walle_project"]
            if walle not in res_pre:
                res_pre[walle] = {
                    "unavailable_hosts": [row["unavailable_hosts"]],
                    "total_hosts": row["total_hosts"]
                }
            else:
                res_pre[walle]["unavailable_hosts"].append(row["unavailable_hosts"])

    res = []
    for walle, aggreg in res_pre.items():
        avg = sum(aggreg["unavailable_hosts"]) / len(aggreg["unavailable_hosts"])
        res.append({
            "unavailable_hosts": avg,
            "total_hosts": aggreg["total_hosts"],
            "unavailable_rate": safe_div(avg, aggreg["total_hosts"]),
            "walle_project": walle,
            "date": week_ago
        })
    return res


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--local_run", type=utils.str2bool, default=False)
    args = parser.parse_args()

    if args.local_run:
        input_file = open("input.json", "r")
        output_file_1 = open("output1.json", "w")
        output_file_2 = open("output2.json", "w")
    else:
        job_context = nv.context()
        inputs = job_context.get_inputs()
        outputs = job_context.get_outputs()
        input_file = open(inputs.get("input"), "r")
        output_file_1 = open(outputs.get("output1"), "w")
        output_file_2 = open(outputs.get("output2"), "w")

    rows = json.load(input_file)
    json.dump(get_last_week_data(rows), output_file_1, indent = 4)
    json.dump(aggregate_week_ago(rows), output_file_2, indent = 4)
        
    


