import json
import argparse
import datetime
import os
import requests

from yp.client import YpClient, find_token
from yp.common import YP_PRODUCTION_CLUSTERS
import nirvana.job_context as nv

import infra.analytics.io_limits_pipeline.utils as utils


def get_abc_slug(abc_id):
    _headers = {'Authorization': 'OAuth {}'.format(os.environ.get("ABC_TOKEN"))}
    url = "https://abc-back.yandex-team.ru/api/v4/services/{0}/?fields=name".format(abc_id)

    result = requests.get(url, headers=_headers).json()

    if result["name"].get("ru"):
        return result["name"]["ru"]
    else:
        return result["name"]["en"]


def get_net_quotas(pre_total_bandwidths):
    date = datetime.date.today().strftime("%Y-%m-%d")
    quotas_abc = []
    quotas_date = []
    total_bandwidths = {}
    remainders = {}
    for entry in pre_total_bandwidths:
        total_bandwidths[entry["cluster"]] = entry["total_bandwidth"]
        remainders[entry["cluster"]] = entry["total_bandwidth"]

    for clst in YP_PRODUCTION_CLUSTERS:
        yp_client = YpClient(clst, config=dict(token=find_token()))
        continuation_token = None
        while True:
            limit = 1000
            options = dict(limit=limit)
            if continuation_token is not None:
                options["continuation_token"] = continuation_token
            data = yp_client.select_objects(
                "account",
                options=options,
                selectors=["/meta/id", "/spec"],
                enable_structured_response=True
            )
            continuation_token = data["continuation_token"]
            for account in data["results"]:
                quota = account[1]["value"].get("resource_limits", {}).get("per_segment", {}).get("default", {}).get("network", {}).get("bandwidth", 0) / (1024 * 1024)
                if quota > 0 and account[0]["value"].startswith("abc:service:"):
                    abc = account[0]["value"][len("abc:service:"):]
                    quotas_abc.append({
                        "abc": get_abc_slug(abc),
                        "cluster": clst,
                        "quota": quota
                    })
                    remainders[clst] -= quota
            if len(data["results"]) < limit:
                break
        quotas_abc.append({
            "abc": "Остаток",
            "cluster": clst,
            "quota": remainders[clst]
        })
        quotas_date.append({
            "cluster": clst,
            "total_quota": total_bandwidths[clst] - remainders[clst],
            "bandwidth_use": (total_bandwidths[clst] - remainders[clst]) / total_bandwidths[clst],
            "total_bandwidth": total_bandwidths[clst],
            "date": date
        })

    return quotas_abc, quotas_date
    

if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--local_run", type=utils.str2bool, default=False)
    args = parser.parse_args()

    if args.local_run is True:
        fout1 = open("out1.json", "w")
        fout2 = open("out2.json", "w")
        fin = open("in.json", "r")
    else:
        job_context = nv.context()
        inputs = job_context.get_inputs()
        outputs = job_context.get_outputs()
        fout1 = open(outputs.get("output1"), "w")
        fout2 = open(outputs.get("output2"), "w")
        fin = open(inputs.get("input"), "r")

    j1, j2 = get_net_quotas(json.load(fin))
    json.dump(j1, fout1, indent="\t", sort_keys=True)
    json.dump(j2, fout2, indent="\t", sort_keys=True)

