import time
import requests
import json
import sys
import os

from yp.client import YpClient

from infra.rtc.analytics.gpu_metrics_collect.utils import func_with_retry

CHUNK_LEN_DEFAULT = 32
HOST_ITYPE = "runtimecloud"


def collect_large_filter(yp_client, a_filter):
    continuation_token = None
    while True:
        limit = 100
        options = dict(limit=limit)
        if continuation_token is not None:
            options["continuation_token"] = continuation_token

        response = yp_client.select_objects(
            "pod",
            options=options,
            enable_structured_response=True,
            selectors=["/meta/id", "/spec", "/labels", "/meta/pod_set_id"], 
            filter=a_filter
        )
        continuation_token = response["continuation_token"]

        for value in response["results"]:
            if value[1]["value"]["node_id"] in default_nodes:
                yield [i["value"] for i in value] + [clst, abc_id]

        if len(response["results"]) < limit:
            break


def get_default_nodes(yp_client):
    return set([str(i[0]["value"]) for i in yp_client.select_objects(
        "node",
        selectors=["/meta/id"],
        filter='[/labels/segment]="default"',
        enable_structured_response=True
    )["results"]])


def get_all_pods_from_abc(abc_id, clst, yp_token):
    yp_client = YpClient(clst, config=dict(token=yp_token))
    default_nodes = get_default_nodes(yp_client)
    account_filter = "[/meta/account_id]='abc:service:{0}'".format(abc_id)
    pod_sets = yp_client.select_objects("pod_set", selectors=["/meta/id"], filter=account_filter)
    if not pod_sets:
        return []
    pods = []
    chunk_len = CHUNK_LEN_DEFAULT

    while pod_sets:
        try:
            pod_set_filter = "or".join(["[/meta/pod_set_id]='{0}'".format(sett[0]) for sett in pod_sets[-chunk_len:]])
            print(pod_set_filter, file=sys.stderr)
            pods_chunk = yp_client.select_objects(
                "pod", 
                selectors=["/meta/id", "/spec", "/labels", "/meta/pod_set_id"],
                filter=pod_set_filter
            )
            pods_chunk = [pod for pod in pods_chunk if pod[1]["node_id"] in default_nodes]
            for i in range(len(pods_chunk)):
                pods_chunk[i] += [clst, abc_id]
            for item in pods_chunk:
                yield item
            pod_sets = pod_sets[:-chunk_len]
            chunk_len = CHUNK_LEN_DEFAULT

        except Exception as e:
            print(e, file=sys.stderr)
            if chunk_len == 1:
                print("Data from pod_set {0} is too heavy".format(pod_sets[-1]), file=sys.stderr)
                collect_large_filter(yp_client, pod_set_filter)
                pod_sets = pod_sets[:-1]
                chunk_len = CHUNK_LEN_DEFAULT
            else:
                chunk_len = int(chunk_len / 2)


def get_all_pods_from_service(service, clst, yp_token):
    yp_client = YpClient(clst, config=dict(token=yp_token))
    default_nodes = get_default_nodes(yp_client)
    pod_filter = "[/labels/deploy/stage_id] = '{0}' or [/meta/pod_set_id] = '{0}' \
        or [/labels/nanny_service_id] = '{0}'".format(service)
    try:
        pods = yp_client.select_objects(
            "pod", 
            selectors=["/meta/id", "/spec", "/labels", "/meta/pod_set_id"], 
            filter=pod_filter
        )
        pods = [pod for pod in pods if pod[1]["node_id"] in default_nodes]

    except Exception as e:
        pods = []
        print(e, file=sys.stderr)
        print("Data from service {0} is too heavy".format(service), file=sys.stderr)
        collect_large_filter(yp_client, pod_filter)

    if not pods:
        return []

    abc_filter = "[/meta/id] = '{0}'".format(pods[0][3])
    try:
        abc_id = yp_client.select_objects(
            "pod_set", 
            selectors=["/meta/account_id"], 
            filter=abc_filter)[0][0].split(":")[2]
    except:
        return []
    for i in range(len(pods)):
        pods[i] += [clst, abc_id]
    return pods


def get_pod_info(pod, clst, yp_token):
    yp_client = YpClient(clst, config=dict(token=yp_token))
    pod_data = yp_client.select_objects(
        "pod", 
        selectors=["/spec", "/labels", "/meta/pod_set_id"], 
        filter="[/meta/id]={0}".format(pod)
    )[0]
    abc = yp_client.select_objects(
        "pod_set", 
        selectors=["/meta/account_id"], 
        filter="[/meta/id]={0}".format(pod_data[3])
    )[0][0]
    return [pod, pod_data[0], pod_data[1], pod_data[2], clst, abc]


def get_walle_hosts(prj):
    url = "https://api.wall-e.yandex-team.ru/v1/hosts?project={0}&limit=10000".format(prj)
    return [a["name"] for a in requests.get(url).json()["result"]]


def get_host_tags(host, sel_tags, prj=None):
    if not sel_tags:
        ret = {"host": host, "itype": HOST_ITYPE}
        if prj:
            ret["prj"] = prj
        return [ret]
    sel_tags = set(sel_tags + ["itype"])
    url = "https://yasm.yandex-team.ru/metainfo/tags/?hosts={0}&tag_format=unified".format(host)
    req = func_with_retry(lambda: requests.get(url)).json()
    tags_list = [{
        tag.split("=")[0]: tag.split("=")[1] for tag in line.split(";") if tag.split("=")[0] in sel_tags
    } for line in req["response"]["result"]]
    for tags in tags_list:
        tags["host"] = host
        if prj != None:
            tags["prj"] = prj
    tags_list = list(json.loads(inst) for inst in set(json.dumps(tags) for tags in tags_list))
    return [tags for tags in tags_list if tags["itype"] == HOST_ITYPE and sel_tags <= set(tags.keys())]


def get_all_abcs():
    url = "https://abc-back.yandex-team.ru/api/v4/services/"
    fin = []
    while url:
        data = requests.get(url, headers={"Authorization": "OAuth {}".format(os.environ.get("ABC_TOKEN"))}).json()
        url = data["next"]
        fin += [{"abc": str(entry["id"])} for entry in data["results"] if entry["state"] not in ["closed", "deleted"]]
    return fin


def get_all_rtc_walles():
    return [{
        "prj": x["id"]
    } for x in requests.get("https://api.wall-e.yandex-team.ru/v1/projects?fields=id&tags=rtc").json()["result"]]


def construct_request(host, collectibles, use):
    base_url = "https://solomon.yandex-team.ru/api/v2/projects/yasm_{0}/sensors/data"
    expression_template = '''signal="{0}", host="{1}", cluster="host_*"'''
    hdrs = {
        "Authorization": "OAuth {}".format(collectibles["token"]),
        "Content-Type": "application/json;charset=UTF-8",
        "Accept": "application/json"
    }

    url = base_url.format(collectibles["itype"])
    expr = expression_template.format(collectibles["signals"], host)

    for tg in use:
        if tg in collectibles and collectibles[tg] != "":
            expr += ''', {0}="{1}"'''.format(tg, collectibles[tg])

    expr = "{" + expr + "}"

    if "series" in collectibles and collectibles["series"] != "":
        expr = collectibles["series"] + "(" + expr + ")"

    if "summary" in collectibles and collectibles["summary"] != "":
        endgoal = collectibles["summary"].split(",")
        endgoal.append(expr)
        expr = endgoal[0] + "(" + ",".join(endgoal[1:]) + ")"

    dowsampling = {"gridMillis": collectibles["gridMillis"], "aggregation": "SUM", "disabled": collectibles["disableDownsampling"]}

    payload = {
        "program": expr,
        "from": collectibles["period"][0],
        "to": collectibles["period"][1],
        "downsampling": dowsampling,
        "version": "HISTOGRAM_FUNCTIONS_DONT_MERGE_3"
    }

    try:
        reslt = requests.post(url, data=json.dumps(payload), headers=hdrs)
        res_json = reslt.json()
    except:
        print("can't recieve data for pod/host {0}, trying again".format(host), file=sys.stderr)
        time.sleep(15)
        try:
            reslt = requests.post(url, data=json.dumps(payload), headers=hdrs)
            res_json = reslt.json()
        except:
            return "Cannot recieve data", reslt.text

    if type(res_json) is dict:
        if "vector" in res_json:
            vect = res_json["vector"]
        elif "timeseries" in res_json:
            vect = [res_json]
        else:
            return "Unknown format was returned", res_json
    elif type(res_json) is list:
        vect = res_json
    else:
        return "Unknown format was returned", res_json

    if len(vect) == 0:
        return "Recieved no data", [url, payload]

    return "OK", vect
