import argparse
import json
import datetime
import os
import requests
import re
import time

import nirvana.job_context as nv
from yp.client import YpClient, find_token
from yp.common import YP_PRODUCTION_CLUSTERS, YP_TESTING_CLUSTERS, YP_PRESTABLE_CLUSTERS

clusters = {
    "production": YP_PRODUCTION_CLUSTERS,
    "testing": YP_TESTING_CLUSTERS,
    "prestable": YP_PRESTABLE_CLUSTERS
}

from infra.analytics.io_limits_pipeline.utils import str2bool


def get_default_nodes(yp_client):
   return set([str(i[0]["value"]) for i in yp_client.select_objects("node",  selectors=["/meta/id"],
                                                                    filter='[/labels/segment]="default"',
                                                                    enable_structured_response=True)["results"]])


def main():
    balancerlist = get_awacs_balancers()
    my_json = []
    for group, cluster_list in clusters.items():
        nannylist = []
        for cluster in cluster_list:
            try:
                with YpClient(cluster, config=dict(token=find_token())) as yp_client:
                    default_nodes = get_default_nodes(yp_client)
                    pods = yp_client.select_objects("pod", selectors=[  "/labels/nanny_service_id",
                                                                        "/spec/host_infra/monitoring/labels/deploy_unit",
                                                                        "/spec/host_infra/monitoring/labels/stage",
                                                                        "/spec/node_id"],
                                                            enable_structured_response=True)["results"]
                    nannylist += [str(i[0]["value"]) for i in pods if str(i[3]["value"]) in default_nodes]
            except:
                print("unable to get data from cluster {0}".format(cluster))
                pass
        my_json.append({"cluster_group": group, "resource": "Nanny services", "value": len(set(nannylist)), "date": datetime.date.today().strftime("%Y-%m-%d")})
        if group == "production":
            my_json.append({"cluster_group": group, "resource": "Awacs balancers", "value": len(set(balancerlist)), "date": datetime.date.today().strftime("%Y-%m-%d")})

    return my_json


def get_all_abcs():
    session = requests.Session()
    session.headers['Content-Type'] = 'application/json'
    session.headers['Authorization'] = 'OAuth {0}'.format(os.getenv('ABC_TOKEN'))
    url = "https://abc-back.yandex-team.ru/api/v4/services/?state__in=develop,supported,needinfo"
    abcs = []
    while url:
        response = session.get(url)
        response_data = response.json()
        url = response_data["next"]
        abcs += [entry["id"] for entry in response_data["results"]]
    return abcs


def get_awacs_balancers():
    session = requests.Session()
    session.headers['Content-Type'] = 'application/json'
    session.headers['Authorization'] = 'OAuth {0}'.format(os.getenv('AWACS_TOKEN'))

    abcs = get_all_abcs()
    namespaces = []

    for abc in abcs:
        response = session.post(
            'https://awacs.yandex-team.ru/api/ListNamespaces/',
            json={"fieldMask": "meta.id",
                "query": {
                    "abcServiceIdIn": [abc]
                }},
        )
        response_data = response.json()
        if response_data.get("namespaces"):
            namespaces += [item['meta']['id'] for item in response_data['namespaces']]
    namespaces = list(set(namespaces))
    print(len(namespaces))

    balancers = []
    for nmspc in namespaces:
        response = session.post(
            'https://awacs.yandex-team.ru/api/ListBalancers/',
            json={'namespaceId': nmspc},
        )
        response_data = response.json()
        balancers += [item['meta']['id'] for item in response_data['balancers']]

    return balancers


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument("--local_run", type=str2bool, default=False)
    args = parser.parse_args()

    if args.local_run is True:
        output_file = "out.json"
    else:
        job_context = nv.context()
        inputs = job_context.get_inputs()
        outputs = job_context.get_outputs()

        output_file = outputs.get("output_file")

    my_json = main()

    with open(output_file, "w") as write_file:
        json.dump(my_json, write_file, indent="\t", sort_keys=True)
