import requests
import logging
from datetime import date, timedelta
from scipy.stats import poisson as pois
import yt.wrapper as yt
import os


class CredentialNotFound(Exception):
    pass


def choose_credential(value, envvar, path):
    """
    Выбирает аутентификационные данные, в порядке приоритета:
         либо явно заданные,
         либо из переменной окружения,
         либо из заданного файла
    :param str value: Заданное пользователем значение
    :param str envvar: Переменная окружения, в которую может быть записано значение
    :param str path: Путь, в котором может лежать значение
    :rtype: str
    :return: Выбранное значение
    """
    if value is None:
        value = os.getenv(envvar)  # В нирвана операциях используется так

    path = os.path.expanduser(path)
    if value is None and os.path.exists(path):
        value = open(path).read().strip()  # Можно использовать для локальных тестов

    if value is None:
        msg = 'credential not found in "{}" nor "{}"'.format(envvar, path)
        raise CredentialNotFound(msg)
    else:
        return value


def execute_query(query, cluster, alias, token, timeout=600):
    proxy = "http://{}.yt.yandex.net".format(cluster)
    s = requests.Session()
    url = "{proxy}/query?database={alias}&password={token}".format(proxy=proxy, alias=alias, token=token)
    resp = s.post(url, data=query, timeout=timeout)
    resp.raise_for_status()
    rows = resp.content.decode("utf-8").strip().split('\n')
    return rows


def get_projects(tag, token):
    adr = "https://api.wall-e.yandex-team.ru/v1/projects?tags={}&fields=id".format(tag)
    req = requests.get(adr, headers={'Authorization': "OAuth " + token})
    staff = req.json()
    return [req['id'] for req in staff['result']]


def count_mu(total, hour, cnt_days):
    return (total*hour)/(24*cnt_days)


def make_pois_dict(name, total, date, project, hrs, cnt_days):
    ans = {'failure': name, 'fielddate': date, 'project': project}
    for hour in hrs:
        ans[str(hour)] = pois.ppf(0.99, count_mu(total, hour, cnt_days))
    return ans


def make_flat_dict(name, total, date, project, hrs, cnt_days):
    ans = {'failure': name, 'fielddate': date, 'project': project}
    for hour in hrs:
        ans[str(hour)] = count_mu(total, hour, cnt_days)
    return ans


def add2dict(pr, ft, count_dict):
    if not count_dict.get(pr):
        count_dict[pr] = {'all': 1}
        count_dict[pr][ft] = 1
    else:
        count_dict[pr]['all'] += 1
        if not count_dict[pr].get(ft):
            count_dict[pr][ft] = 1
        else:
            count_dict[pr][ft] += 1


def main():
    cluster = 'hahn'
    alias = "*ch_public"
    token = choose_credential(None, 'YT_TOKEN', '~/.yt/token')

    proj_yt = get_projects('yt', token)
    proj_rtc = get_projects('rtc', token)

    name_yt = 'tag_yt'
    name_rtc = 'tag_rtc'
    name_all = 'all'

    hours = [1, 6, 12, 24, 24*7, 24*28]
    time_dict = {}

    i_date = date.today() - timedelta(days=1)
    count_days = 0

    yt.config.set_proxy("hahn")

    while True:
        try:
            for i in range(365):
                if str(i_date) == '2019-01-08':
                    break
                else:
                    count_days += 1
                query = "SELECT failure_timestamp, hostname, project, failure_type " \
                        "FROM `//home/logfeller/logs/walle-failures-log/1d/{}`".format(str(i_date))
                result = execute_query(query=query, cluster=cluster, alias=alias, token=token)
                for row in result:
                    ts, hn, pr, ft = row.split()
                    if time_dict.get((ts, hn)):
                        if time_dict[(ts, hn)] != (pr, ft):
                            raise Exception("Reiteration of timestamp {} and host name {}".format(str(ts), hn))
                    else:
                        time_dict[(ts, hn)] = (pr, ft)
                i_date -= timedelta(days=1)
            break
        except:
            i_date -= timedelta(days=1)
            continue

    count_dict = {}

    for val in time_dict.values():
        pr, ft = val
        add2dict(pr, ft, count_dict)
        add2dict(name_all, ft, count_dict)
        if pr in proj_rtc:
            add2dict(name_rtc, ft, count_dict)
        if pr in proj_yt:
            add2dict(name_yt, ft, count_dict)

    yt.config.set_proxy("hahn")
    today = str(date.today())

    tp_pois = yt.TablePath("//home/walle/failrate/poisson_99", append=True)
    tp_flat = yt.TablePath("//home/walle/failrate/flat", append=True)

    for project in count_dict.keys():

        yt.write_table(tp_pois,
                       [make_pois_dict(key, count_dict[project][key], today, project, hours, count_days) for key in
                        count_dict[project].keys()])

        yt.write_table(tp_flat,
                       [make_flat_dict(key, count_dict[project][key], today, project, hours, count_days) for key in
                        count_dict[project].keys()])


if __name__ == '__main__':
    main()
