import statistics
import numpy as np
import math
import sys
from pprint import pprint
from yp.client import YpClient, find_token
from yp.common import YP_CLUSTERS
from yql.api.v1.client import YqlClient


YP_CLIENTS = {
    "iva": YpClient("iva", config=dict(token=find_token())),
    "vla": YpClient("vla", config=dict(token=find_token())),
    "man": YpClient("man", config=dict(token=find_token())),
    "sas": YpClient("sas", config=dict(token=find_token())),
    "myt": YpClient("myt", config=dict(token=find_token()))

}

YP_CLUSTERS.remove("pre")


INPUT_TO_MAPPER_TYPE = {
    "walles": "obj",
    "abcs": "abc",
    "services": "service",
    "pods": "pod",
    "pods_cached": "pod",
    "hosts": "host",
    "hosts_cached": "host"
}


def is_histogram(signals):
    sg = signals.split("||")[0]
    return sg.endswith("hgram") or sg.endswith("hhh")


def list_to_histogram(lst, max_size):
    np_lst = np.array(lst)

    zeroes = len(np_lst[np_lst == 0])
    np_lst = np_lst[np_lst != 0]

    np_lst = np.sort(np_lst)
    np_hst = np.histogram(np_lst, bins="auto")

    size = sys.getsizeof(np_hst)
    if size > max_size:
        true_size = int(len(np_hst[0]) * max_size / size)
        np_hst = np.histogram(np_lst, bins=true_size)

    return {
        "zeroes": zeroes,
        "edges": np_hst[1].tolist(),
        "counts": np_hst[0].tolist()
    }


def get_summary(signal):
    infix = signal[-3:-1]
    wrapmap = {
        "nn": "summary_min",
        "xx": "summary_max",
        "mm": "summary_sum",
        "ee": "summary_sum",
        "um": "summary_sum", #_summ
        "tt": "summary_last",
        "vv": "summary_avg"
    }
    return wrapmap.get(infix, "")


def get_series(signal):
    infix = signal[-3:-1]
    wrapmap = {
        "nn": "series_min",
        "xx": "series_max",
        "mm": "series_sum",
        "ee": "series_sum",
        "um": "series_sum", #_summ
        "tt": "series_avg",
        "vv": "series_avg",
        "hh": "series_sum",
        "ra": "series_sum", #_hgram
    }
    return wrapmap.get(infix, "")


def get_histogram_func(func):
    if func.startswith("percentile"):
        percentiles = func[len("percentile"):].strip("()").split(",")
        if len(percentiles) > 1:
            return "histogram_percentile,as_vector(" + ",".join(percentiles) + ")"
        else:
            return "histogram_percentile," + percentiles[0] + ")"

    funcmap = {
        "sum": "histogram_sum",
        "avg": "histogram_avg",
        "max": "histogram_percentile,100",
        "min": "histogram_percentile,0",
        "median": "histogram_percentile,50",
        "hist": ""
    }
    if func not in funcmap:
        raise Exception("Unknown aggregator: {0}".format(func))
    return funcmap[func]


def filter_nans(lst):
    for i in range(len(lst)):
        if type(lst[i]) not in [float, int] or math.isnan(lst[i]):
            lst[i] = 0.0


def get_list_func(lst, func, max_size):
    filter_nans(lst)

    if func.startswith("percentile"):
        percentiles = [float(num.strip()) for num in func.replace("percentile", "").strip("()").split(",")]
        res = np.percentile(lst, percentiles).tolist()
        return {
            "percentile_" + str(percentiles[i]): res[i] for i in range(len(percentiles))
        }

    funcmap = {
        "hist": lambda x: list_to_histogram(x, max_size),
        "avg": statistics.mean,
        "median": statistics.median,
        "mode": statistics.mode,
        "sum": sum,
        "max": max,
        "min": min
    }
    if func not in funcmap:
        raise Exception("Unknown aggregator: {0}".format(func))

    return {func: funcmap[func](lst)}


def extract_deploy_units():
    deploy_services = {}

    client = YqlClient(db='hahn')

    query = '''
    SELECT distinct cgrp
    from hahn.`home/runtimecloud/research/KERNEL-691/cgrps_resolved`
    where cgrp like '%.%'
    '''

    request = client.query(
        query,
        syntax_version=1
    )
    request.run()

    for table in request.get_results():
        table.fetch_full_data()

        for row in table.rows:
            deploy_services[row[0]] = row[0]

    return deploy_services


def extract_podsets_without_guarantees():
    pod_sets = set()

    for cluster in YP_CLUSTERS:
        with YpClient(cluster, config=dict(token=find_token())) as yp_client:
            podsets = yp_client.select_objects(
                "pod", selectors=["/meta/pod_set_id"],
                filter="[/spec/resource_requests/network_bandwidth_guarantee]=0u", enable_structured_response=True
            )

            for record in podsets["results"]:
                pod_sets.add(record[0]["value"])

    return pod_sets


def extract_all_services():

    nanny_services_to_process = {}

    for cluster in YP_CLUSTERS:
        with YpClient(cluster, config=dict(token=find_token())) as yp_client:
            services = yp_client.select_objects(
                "pod_set", selectors=["/spec/pod_disruption_budget_id", "/meta/id"],
                filter="[/spec/pod_disruption_budget_id]!=''", enable_structured_response=True
            )

            for record in services["results"]:
                try:
                    nanny_services_to_process[record[0]["value"]] = record[1]["value"]
                except TypeError:
                    pass

    deploy_services = extract_deploy_units()
    total_services = {**nanny_services_to_process, **deploy_services}
    podsets_without_guarantees = extract_podsets_without_guarantees()
    services_without_guarantee = list(set([k for k, v in total_services.items() if v in podsets_without_guarantees]))

    return services_without_guarantee
