"""Host certificate control tool."""

import getpass
import logging
import logging.handlers
import os
import pprint
import requests
import socket
import sys
import time

from asn1crypto import crl
from asn1crypto import pem
from asn1crypto import x509

from library.python import svn_version

from infra.rtc.certman import certificate, config, fileutil, jugglerutil, yasmutil

# https://oauth.yandex-team.ru/client/d5dc1a8036814ca2920b84b1232da76e
CLIENT_ID = "d5dc1a8036814ca2920b84b1232da76e"
CLIENT_SECRET = "d51ec009e65a49eb905c21255935729e"
# Acquired by manually filling in fields in IDM
IDM_REQ_URL = ('https://idm.yandex-team.ru/user/{login}/roles'
               '#rf=1,rf-role=1cQ7DCpy#{login}@certificator/'
               'group-4;;10;'
               '%D0%90%D0%B2%D0%B0%D1%80%D0%B8%D0%B9%D0%BD%D1%8B%D0%B9%20'
               '%D0%BF%D0%B5%D1%80%D0%B5%D0%B2%D1%8B%D0%BF%D1%83%D1%81%D0'
               '%BA%20%D1%81%D0%B5%D1%80%D1%82%D0%B8%D1%84%D0%B8%D0%BA%D0%B0%D1%82%D0%BE%D0%B2%20%D0%B2%20RTC,'
               'f-status=all,sort-by=-updated,rf-expanded=1cQ7DCpy')
REQUEST_CERT_URL = 'https://crt-api.yandex-team.ru/api/v2/certificates/'

log = logging.getLogger(__name__)


def init_log_file(log_file, log_format=None, max_bytes=8388608, backup_count=7):
    file_handler = logging.handlers.RotatingFileHandler(
        log_file,
        maxBytes=max_bytes,
        backupCount=backup_count,
    )
    file_handler.setFormatter(logging.Formatter(
        log_format or '[%(asctime)s %(process)d] %(levelname)s %(message)s'))
    logging.getLogger().addHandler(file_handler)


def get_cauth_token(login):
    log.debug(f'Acquiring token for user {login}')
    from library.python import oauth
    return oauth.get_token(CLIENT_ID, CLIENT_SECRET, login=login)


def gen_token(login):
    token = get_cauth_token(login)
    if not token:
        log.error(f'Failed to obtain token for user {login}')
        return 1
    print('Token:', token)


def issue(token, hostname, dest, login):
    """Issue completely new certificate (existing cert is not required)"""
    if not token:
        log.info('No token provided, will try to acquire one...')
        token = get_cauth_token(login)

    if not token:
        log.error('Failed to obtain token. Please provide one via --token or check SSH_AGENT/keys.')
        return 1

    headers = {
        'Authorization': 'OAuth {}'.format(token),
    }
    data = {
        'ca_name': 'RcInternalCA',
        'type': 'rc-server',
        'common_name': hostname,
    }
    resp = requests.post(REQUEST_CERT_URL, json=data, headers=headers)
    if resp.status_code != 201:
        content = resp.content.decode('utf8')
        log.error(f'Bad response from CRT: {content}')
        if 'You do not have permission to perform this action' in content:
            log.error('Please, request role in IDM: %s', IDM_REQ_URL.format(login=login))
        return 1

    download_url = resp.json()['download'] + '?format=pem'
    resp = requests.get(download_url, headers=headers)
    if resp.status_code != 200:
        log.error('Bad response from download url')
        return 1

    try:
        certificate.save_pem(resp.content, dest)
    except Exception as e:
        log.error(f'Failed to save PEM: {e}')
        return 1

    log.info(f'Certificate successfully issued and saved to {dest}')


def info(f):
    pem_bytes = f.read()
    try:
        it = pem.unarmor(pem_bytes, multiple=True)
    except Exception as e:
        log.error(f'Failed to parse PEM bytes: {e}')
        return 1

    # Iterate until first CERTIFICATE
    c = None
    while 1:
        try:
            type_name, headers, der_bytes = next(it)
        except StopIteration:
            break
        except Exception as e:
            log.error(f'Failed to decode PEM: {e}')
            return 1
        if type_name != 'CERTIFICATE':
            continue
        try:
            c = x509.Certificate.load(der_bytes)
        except Exception as e:
            log.error(f'Failed to load DER from {type_name} PEM: {e}')
            return 1
        break

    if c is None:
        log.error('No one certificate found in pem file')
        return 1

    print('First certificate in PEM:')
    print('-' * 40)
    pprint.pprint(c.native, indent=4)
    print('-' * 40)


def crl_cmd(f):
    cert_list = crl.CertificateList.load(f.read())
    print('Serial numbers:')
    for revoked_cert in cert_list['tbs_cert_list']['revoked_certificates']:
        print('  *', revoked_cert['user_certificate'].native)


def status_cmd(conf):
    try:
        status = certificate.Status.load_from_file(conf.state_file)
    except Exception as e:
        if conf.juggler:
            print(jugglerutil.fmt_check_text('certman', 'WARN', str(e)))
            return 0

        log.fatal(f'Failed to load state: {e}')
        return 2

    is_state_outdated = time.time() - status.mtime > 86400  # 24 hours
    is_valid = status.is_valid()
    is_update_required = status.is_update_required(
        certificate.get_current_utc_tz_aware_datetime(),
        conf.cert_min_days,
        conf.cert_max_days,
        socket.gethostname(),
    )

    if conf.juggler:
        # Be careful with CRITs: used as triggers for Wall-e to redeploy hosts!
        if is_valid:
            if is_state_outdated:
                txt = jugglerutil.fmt_check_text('certman', 'WARN', 'State outdated')
            elif is_update_required:
                txt = jugglerutil.fmt_check_text('certman', 'WARN', 'Update required')
            else:
                txt = jugglerutil.fmt_check_text('certman', 'OK', 'Ok')
        else:
            txt = jugglerutil.fmt_check_text('certman', 'CRIT', status.error_msg)

        print(txt)
        return 0

    if is_state_outdated:
        print(f'State outdated: {is_state_outdated}')
        return 8

    print(f'Valid: {is_valid}')
    print(f'Update required: {is_update_required}')

    if is_valid:
        print(f'Days left: {status.days_left}')
        return 0

    print(f'Error: {status.error_msg}')
    return 4


def push_yasm_metrics(conf, status):
    log.debug('Sending YASM metrics')
    yasm_signals = (
        ('hostman-cert-days-left_thhh', status.days_left),
        ('hostman-cert-valid_thhh', int(status.is_valid())),
        ('hostman-cert-needs-update_thhh', int(status.is_update_required(
            certificate.get_current_utc_tz_aware_datetime(),
            conf.cert_min_days,
            conf.cert_max_days,
            socket.gethostname(),
        ))),
    )
    err = yasmutil.push_yasm_metrics(
        yasm_signals,
        tags={
            'ctype': conf.yasm_metrics_ctype,
            'itype': conf.yasm_metrics_itype,
            'prj': conf.yasm_metrics_prj,
        },
        ttl=conf.yasm_metrics_ttl,
    )

    if err is not None:
        log.error(err)


def run(args=None):
    conf = config.get_conf(args=args, desc=__doc__)
    changing_commands = {'manage', 'issue'}

    # All changing commands MUST be logged to file (debug and security reasons)
    if conf.cmd in changing_commands:
        init_log_file(conf.log_file)

    user = getpass.getuser()
    sudo_user = os.getenv('SUDO_USER')
    revision = svn_version.svn_revision()

    log.debug('Started r{revision} by {user}{sudo_user} as {argv}'.format(
        revision=revision,
        user=user,
        sudo_user=f' (by {sudo_user} using sudo)' if sudo_user else '',
        argv=str(sys.argv),
    ))

    try:
        if conf.cmd in changing_commands:
            # All changing commands MUST be performed using file lock (rest are
            # just fine when all files atomically written). Deliberately don't
            # release lock to assure it exists until exit.
            try:
                # keep file object in variable, otherwise GC will steal it
                f = fileutil.lock_file(conf.lock_file)  # noqa F841
            except BlockingIOError:
                log.error(f'Failed to lock {conf.lock_file} (already locked)')
                return 2

        if conf.cmd == 'status':
            return status_cmd(conf)

        if conf.cmd == 'manage':
            status = certificate.Certificate(socket.gethostname()).manage(
                conf.pem_file,
                conf.cert_min_days,
                conf.cert_max_days,
                conf.crl_file,
                conf.dry_run,
            )
            status.dump_to_file(conf.state_file)

            if conf.push_yasm_metrics:
                push_yasm_metrics(conf, status)

            if status.is_valid():
                return 0

            log.error(status.error_msg)
            return 4

        if conf.cmd == 'crl':
            with open(conf.crl_file, 'rb') as f:
                return crl_cmd(f)

        if conf.cmd == 'gen-token':
            return gen_token(conf.user or sudo_user or user)

        if conf.cmd == 'info':
            try:
                with open(conf.pem_file, 'rb') as f:
                    return info(f)
            except PermissionError as e:
                log.error(str(e))
                log.info('Try sudo {}'.format(' '.join(sys.argv)))

        elif conf.cmd == 'issue':
            return issue(conf.token, conf.hostname, conf.pem_file, sudo_user or user)

    except BaseException:  # to catch KeyboardInterrupt as well
        log.fatal('Exception raised:', exc_info=True)

    return 127  # Unknown error
