import argparse
import json
import logging
import socket

from library.python.svn_version import svn_revision


DEFAULT_CONF_FILE = '/etc/certctl.conf'
DEFAULT_OPTS = {
    'cert_max_days': 365,
    'cert_min_days': 7,
    'crl_file': '/var/lib/certctl/crl',
    'lock_file': '/var/run/certctl.lock',
    'log_file': '/var/log/certctl.log',
    'pem_file': '/etc/certs/capi.pem',
    'state_file': '/var/lib/certctl/state',
    'yasm_metrics_ctype': 'production',
    'yasm_metrics_itype': 'runtimecloud',
    'yasm_metrics_prj': 'none',
    'yasm_metrics_ttl': 3600 * 2,
}

log = logging.getLogger(__name__)
logging.basicConfig(format='%(asctime)s %(levelname)s %(message)s',
                    level=logging.DEBUG)


def get_conf(args=None, desc=None):
    conf = DEFAULT_OPTS.copy()

    # Just cli args are not enough: host cpecific opts (prestable with
    # different TTL etc) should be preserved for manual runs.
    argparser = argparse.ArgumentParser(add_help=False)
    argparser.add_argument(
        '--conf-file',
        help='default is "{}"'.format(DEFAULT_CONF_FILE),
        metavar='PATH',
        type=str,
    )
    argparser.add_argument(
        '--version',
        action='version',
        version='%(prog)s r{}'.format(svn_revision()),
        help='print version and exit',
    )

    # Load conf beforehand to have correct defaults for the rest options
    known_args, rest_args = argparser.parse_known_args(args=args)
    if known_args.conf_file is None:
        conf_filename = DEFAULT_CONF_FILE
    else:
        conf_filename = known_args.conf_file

    try:
        with open(conf_filename) as f:
            conf.update(json.load(f))
    except FileNotFoundError as e:
        if known_args.conf_file is not None:
            # User defined conf file MUST exist (defined explicitly), while
            # default conf file MAY absent (defined implicitly).
            log.fatal(f'Unable to load conf file: {e}')
            raise SystemExit(1)
    except Exception as e:  # noqa
        log.fatal(f'Failed to load conf from {conf_filename} ({e})')
        raise SystemExit(1)

    # Main argparser; inherite previous to cover it's opts in help message
    argparser = argparse.ArgumentParser(description=desc, parents=[argparser])
    argparser.add_argument(
        '--log-file',
        default=conf['log_file'],
        help='default is "%(default)s"',
        metavar='PATH',
        type=str,
    )
    argparser.add_argument(
        '--log-level',
        choices={'CRITICAL', 'ERROR', 'WARNING', 'INFO', 'DEBUG'},
        default='DEBUG',
        help='default is "%(default)s"',
        metavar='LEVEL',
        type=set_loglevel,
    )

    cert_max_days_parser = argparse.ArgumentParser(add_help=False)
    cert_max_days_parser.add_argument(
        '--cert-max-days',
        default=conf['cert_max_days'],
        help='TTL max days for new certificate; default is %(default)s',
        metavar='NUM',
        type=int,
    )

    cert_min_days_parser = argparse.ArgumentParser(add_help=False)
    cert_min_days_parser.add_argument(
        '--cert-min-days',
        default=conf['cert_min_days'],
        help='threshold days for certificate EOL check; default is %(default)s',
        metavar='NUM',
        type=int,
    )

    crl_file_parser = argparse.ArgumentParser(add_help=False)
    crl_file_parser.add_argument(
        '--crl-file',
        default=conf['crl_file'],
        help='Certificate Revokation List; default is "%(default)s"',
        metavar='PATH',
        type=str,
    )

    lock_file_parser = argparse.ArgumentParser(add_help=False)
    lock_file_parser.add_argument(
        '--lock-file',
        default=conf['lock_file'],
        help='path to lock file; default is "%(default)s"',
        metavar='PATH',
        type=str,
    )

    pem_file_parser = argparse.ArgumentParser(add_help=False)
    pem_file_parser.add_argument(
        '--pem-file',
        default=conf['pem_file'],
        help='path to certificate (PEM) file; default is "%(default)s"',
        metavar='PATH',
        type=str,
    )

    # State file serve two main things:
    # * provides info for monitorings runned without root priveleges
    # * allows to reliably check CRL (keep check times)
    state_file_parser = argparse.ArgumentParser(add_help=False)
    state_file_parser.add_argument(
        '--state-file',
        default=conf['state_file'],
        help='path to state file; default is "%(default)s"',
        metavar='PATH',
        type=str,
    )

    commands = argparser.add_subparsers(dest='cmd', required=True)

    cmd = commands.add_parser('crl', help='Decode and dump CRL file',
                              parents=[crl_file_parser])

    cmd = commands.add_parser('gen-token', help='Generate OAuth token')
    cmd.add_argument(
        '--user',
        help='default: current uid',
        metavar='NAME',
        type=str,
    )

    cmd = commands.add_parser('info', help='Decode and dump PEM file',
                              parents=[pem_file_parser])

    cmd = commands.add_parser('issue', help='Issue new certificate',
                              parents=[lock_file_parser, pem_file_parser])
    cmd.add_argument(
        '--token',
        help='OAuth token (may be acquired using `gen-token` command)',
        type=str,
    )
    cmd.add_argument(
        '--hostname',
        default=socket.gethostname(),
        help='hostname for certificate, default is current hostname',
        type=str,
    )

    cmd = commands.add_parser(
        'manage',
        help="Issue new certificete if current one near it's end of life",
        parents=[crl_file_parser, lock_file_parser, pem_file_parser,
                 state_file_parser, cert_max_days_parser, cert_min_days_parser],
    )
    cmd.add_argument(
        '--dry-run',
        action='store_true',
        help='do not update certificate, just check it',
    )
    cmd.add_argument(
        '--push-yasm-metrics',
        action='store_true',
        help='push metrics to local yasm client',
    )
    cmd.add_argument(
        '--yasm-metrics-ttl',
        default=conf['yasm_metrics_ttl'],
        help='TTL seconds for yasm metrics, default is "%(default)s"',
        metavar='NUM',
        type=int,
    )
    cmd.add_argument(
        '--yasm-metrics-ctype',
        default=conf['yasm_metrics_ctype'],
        help='`ctype` for YASM metrics, default is "%(default)s"',
        metavar='STR',
        type=str,
    )
    cmd.add_argument(
        '--yasm-metrics-itype',
        default=conf['yasm_metrics_itype'],
        help='`itype` for YASM metrics, default is "%(default)s"',
        metavar='STR',
        type=str,
    )
    cmd.add_argument(
        '--yasm-metrics-prj',
        default=conf['yasm_metrics_prj'],
        help='project (prj) for YASM metrics, default is "%(default)s"',
        metavar='STR',
        type=str,
    )

    cmd = commands.add_parser(
        'status',
        help="Print certificate status",
        parents=[state_file_parser, cert_max_days_parser, cert_min_days_parser],
    )
    cmd.add_argument(
        '--juggler',
        action='store_true',
        help='print status in juggler check format',
    )

    return argparser.parse_args(args=rest_args)


def set_loglevel(level_name):
    logging.getLogger().setLevel(level_name)
    return level_name  # argparser expect value to be returned
