import logging

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

YASM_TIMEOUT = 10.0
YASM_RETRIES = 3
YASM_TTL = 2 * 3600

log = logging.getLogger(__name__)


def make_signals(signal_values, tags=None, ttl=YASM_TTL):
    signals = {'values': []}

    if ttl is not None:
        signals['ttl'] = ttl
        log.debug('Golovan TTL signals {}'.format(signals['ttl']))

    signals['tags'] = tags or {}
    log.debug('Golovan tags {}'.format(signals['tags']))

    for name, value in signal_values:
        log.debug('Signal "{}" value "{}"'.format(name, value))
        signals['values'].append(
            {
                "name": name,
                "val": value,
            }
        )
    return [signals]


def push_yasm_metrics(signal_values, tags=None, ttl=YASM_TTL):
    signals = make_signals(signal_values, tags=tags, ttl=ttl)
    retries = Retry(
        total=YASM_RETRIES,
        read=YASM_RETRIES,
        connect=YASM_RETRIES,
        backoff_factor=0.5,
        status_forcelist=[500, 502, 503, 504],
        method_whitelist=frozenset(['POST'])
    )
    try:
        with requests.Session() as s:
            s.trust_env = False  # Do not use .netrc
            s.mount('http://', HTTPAdapter(max_retries=retries))
            r = s.post("http://[::1]:11005/", json=signals, timeout=YASM_TIMEOUT)
            data = r.json()
    except Exception as e:
        return "exception in request to golovan. Exception '{}'".format(e)
    if r.status_code != 200:
        return "yasm http response code: {}; text: {}".format(r.status_code, r.text)
    if data['status'] == 'ok':
        return None
    return "yasm response status: {}".format(data)
