from infra.rtc.docker_registry.docker_torrents.exceptions import InsufficientParametersException
from infra.rtc.docker_registry.docker_torrents.clients.tvm_tool import TvmToolClient
from infra.rtc.docker_registry.docker_torrents.exceptions import SubrequestException
from infra.rtc.docker_registry.docker_torrents.torrent_config import TorrentsConfig
from infra.rtc.docker_registry.docker_torrents.exceptions import BlackboxException
from requests import HTTPError
import requests


class BlackboxClient:

    def __init__(self, config: TorrentsConfig):
        self.scopes = config.blackbox.get('scopes', None)
        self.url_template = str(config.blackbox['url_template'])
        self.ip_header = str(config.blackbox.get('ip_header', 'X-Real-Ip'))
        self.connect_timeout = float(config.blackbox.get('connect_timeout', 0.1))
        self.read_timeout = float(config.blackbox.get('read_timeout', 1))
        self.tvm_ticket_header = str(config.blackbox.get('tvm_ticket_header', 'X-Ya-Service-Ticket'))
        if config.tvm_tool is not None:
            self.tvm_tool = TvmToolClient(config)
        else:
            self.tvm_tool = None

    def login(self, headers: dict, user_ip: str = None) -> str:
        request_headers = dict()
        request_headers.update(headers)
        if self.ip_header in request_headers:
            user_ip = request_headers[self.ip_header]
        if user_ip is None:
            raise InsufficientParametersException(
                '{} request header or user_ip parameter is mandatory'.format(self.ip_header))
        if self.tvm_tool is not None:
            request_headers[self.tvm_ticket_header] = self.tvm_tool.get_ticket('blackbox')
        if self.scopes is not None:
            url = self.url_template.format(
                payload="method=oauth&format=json&userip={}&scopes={}".format(user_ip, self.scopes)
            )
        else:
            url = self.url_template.format(
                payload="method=oauth&format=json&userip={}".format(user_ip)
            )
        with requests.request(
            method='GET',
            url=url,
            headers=request_headers,
            timeout=(self.connect_timeout, self.read_timeout)
        ) as response:
            try:
                response.raise_for_status()
                result = response.json()
                if 'exception' in result:
                    raise BlackboxException(result['exception']['value'], result['error'])
                return response.json()['login']
            except HTTPError as http_error:
                raise SubrequestException(http_error, response.text)
            except Exception as error:
                raise error
