import logging
from time import sleep
import requests
import random
import json


class IRBTorrentBrewerHelper:
    # abstract torrent brewer class

    def generate_rbtorrent(self, filename: str, mds_key: str) -> str:
        raise NotImplementedError("Mds or alternative torrent interface is not set")

    def remove_rbtorrent(self, filename: str, mds_key: str):
        raise NotImplementedError("Mds or alternative torrent interface is not set")


class ObsoleteTorrentHelper(IRBTorrentBrewerHelper):
    # obsolete torrent helper for mds

    def __init__(self,
                 logger: logging.Logger,
                 url_template=None,
                 torrent_url=None,
                 namespace=None,
                 connect_timeout=0.1,
                 read_timeout=600,
                 retries=3,
                 fallback_timeout=1):
        self.url_template = url_template
        self.torrent_url = torrent_url
        self.namespace = namespace
        self.logger = logger
        self.connect_timeout = connect_timeout
        self.read_timeout = read_timeout
        self.retries = retries
        self.fallback = fallback_timeout

    def generate_rbtorrent(self, filename: str, mds_key: str) -> str:
        return self._announce_request(filename, mds_key)

    def _announce_request(self, filename: str, mds_key: str, remove: bool = False) -> str:
        tries = 0
        while tries <= self.retries:
            try:
                post_data = [{
                    'ns': self.namespace,
                    'mds_key': mds_key,
                    'type': 'file',
                    'path': filename
                }]
                url = self.torrent_url
                if remove:
                    url = '{}?ttl={}'.format(url, '1')
                resp = requests.post(
                    url,
                    data=json.dumps(post_data),
                    timeout=(self.connect_timeout, self.read_timeout)
                )
                resp.raise_for_status()
                return resp.text
            except Exception as error:
                if tries >= self.retries:
                    raise error
                self.logger.exception('Exception during torrent announce {} {}'.format(filename, mds_key))
                timeout = self.fallback * (2 ** tries)
                timeout = timeout + random.uniform(0, timeout / 2)
                sleep(timeout)
            tries += 1

    def remove_rbtorrent(self, filename: str, mds_key: str):
        self._announce_request(filename, mds_key, True)
