from infra.rtc.docker_registry.docker_torrents.exceptions import InsufficientParametersException
from infra.rtc.docker_registry.docker_torrents.exceptions import SubrequestException
from infra.rtc.docker_registry.docker_torrents.torrent_config import TorrentsConfig
from requests import HTTPError
import requests


class RegistryClient:

    def __init__(self, config: TorrentsConfig):
        self.host = str(config.docker['host'])
        self.proto = str(config.docker['proto'])
        self.image_resolve_template = str(config.docker['image_resolve_template'])
        self.hash_resolve_template = str(config.docker['hash_resolve_template'])
        self.tags_resolve_template = str(config.docker['tags_resolve_template'])
        self.blob_sum_prefix = str(config.docker['blob_sum_prefix'])
        self.compatibility_header = str(config.docker['compatibility_header'])
        self.connect_timeout = float(config.docker.get('connect_timeout', 0.1))
        self.read_timeout = float(config.docker.get('read_timeout', 1))

    def resolve_tags(self, scope: str, headers: dict) -> list:
        request_headers = dict()
        request_headers.update(headers)
        if 'Authorization' not in request_headers:
            raise InsufficientParametersException('Authorization header is mandatory')
        url = self.tags_resolve_template.format(
            proto=self.proto,
            host=self.host,
            scope=scope
        )
        response = requests.request(
            method='GET',
            url=url,
            headers=request_headers,
            timeout=(self.connect_timeout, self.read_timeout)
        )
        try:
            response.raise_for_status()
            result = list()
            for tag in response.json()['tags']:
                result.append(tag)
            return result
        except HTTPError as http_error:
            raise SubrequestException(http_error, response.text)
        except Exception as error:
            raise error

    def resolve_manifest(self, scope: str, tag: str, headers: dict) -> dict:
        request_headers = dict()
        request_headers.update(headers)
        if 'Authorization' not in request_headers:
            raise InsufficientParametersException('Authorization header is mandatory')
        url = self.image_resolve_template.format(
            proto=self.proto,
            host=self.host,
            scope=scope,
            tag=tag
        )
        request_headers['Accept'] = self.compatibility_header
        response = requests.request(
            method='GET',
            url=url,
            headers=request_headers,
            timeout=(self.connect_timeout, self.read_timeout)
        )
        try:
            response.raise_for_status()
            return response.json()
        except HTTPError as http_error:
            raise SubrequestException(http_error, response.text)
        except Exception as error:
            raise error
