import logging
import os
import hashlib
import tvmauth
import requests


from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry


from .rbtorrent_helper import IRBTorrentBrewerHelper, ObsoleteTorrentHelper
import infra.skyboned.api


class SkybonedTorrentNotAdded(Exception):
    pass


class SkybonedTorrentHelper(IRBTorrentBrewerHelper):
    # torrent helper based on skyboned api

    def __init__(self,
                 logger: logging.Logger,
                 skyboned_api_servers=None,
                 mds_read_baseurl=None,
                 namespace=None,
                 connect_timeout=0.1,
                 read_timeout=600,
                 retries=3,
                 tvm_id=None,
                 tvm_secret=None,
                 ):
        self.logger = logger
        self.server = skyboned_api_servers

        # mds connect parameters
        self.connect_timeout = connect_timeout
        self.read_timeout = read_timeout
        self.retries = retries
        self.mds_read_baseurl = mds_read_baseurl
        self.namespace = namespace

        # obtain tvm_ticket
        self.tvm_id = tvm_id
        self.tvm_secret = tvm_secret

    @staticmethod
    def get_mds_content_ranges(content_size, chunk_size):
        """
        :param content_size: size of mds content
        :param chunk_size: size of chunk
        """

        ranges = []
        pos = 0

        while content_size > chunk_size:
            ranges.append((pos, pos + chunk_size - 1))
            content_size -= chunk_size
            pos += chunk_size
        ranges.append((pos, pos + content_size - 1))
        return ranges

    def _request_tvm_ticket(self):
        """
        :return: tvm_ticket
        """
        tvm_cache_path = ' /tmp/tvm_skyboned'
        if not os.path.exists(tvm_cache_path):
            os.makedirs(tvm_cache_path, exist_ok=True)
        tvm_client = tvmauth.TvmClient(
            tvmauth.TvmApiClientSettings(
                self_tvm_id=int(self.tvm_id),
                self_secret=self.tvm_secret,
                dsts={'skyboned': 2021848, "datasync": 2000060},
                disk_cache_dir=tvm_cache_path,
            )
        )
        tvm_ticket = tvm_client.get_service_ticket_for('skyboned')

        if not tvm_ticket:
            raise Exception('no tvm ticket obtained for skyboned helper client')
        return tvm_ticket

    def _get_mds_url(self, mds_key):
        """
        :param mds_key: mds key couple
        :return: mds url for content
        """
        return f"{self.mds_read_baseurl}/get-{self.namespace}/{mds_key}"

    def _get_http_session(self):
        """
        :return: mds http session
        """

        retry_strategy = Retry(
            total=self.retries,
            read=self.retries,
            connect=self.retries,
            backoff_factor=0.3,
            status_forcelist=[429, 500, 502, 503, 504],
            method_whitelist=["HEAD", "GET", "OPTIONS"]
        )
        adapter = HTTPAdapter(max_retries=retry_strategy)
        mds_session = requests.Session()
        mds_session.mount(self.mds_read_baseurl, adapter)
        return mds_session

    def _get_mds_content_size(self, stream_url: str):
        try:
            with self._get_http_session() as s:
                r = s.head(stream_url, timeout=self.connect_timeout)
        except Exception as e:
            self.logger.exception("can't get size for %s with exception %s" % (stream_url, e))
            raise Exception(e)
        if r.ok:
            if r.headers.get('Content-Length'):
                return int(r.headers['Content-Length'])
            elif r.headers.get('X-Data-Size'):
                return int(r.headers['X-Data-Size'])
            else:
                raise Exception("can't get size for %s with response headers %s" % (stream_url, r.headers))
        raise Exception("can't get size for %s" % stream_url)

    def _get_mds_content_with_range(self, stream_url: str, stream_range: set):
        """
        :param stream_url: url to mds docker image source
        :param stream_range: range header for mds
        """

        if len(stream_range) != 2:
            self.logger.exception('Wrong range %s for %s' % (stream_range, stream_url))
            raise ValueError('Wrong range %s')

        range_header_value = f"bytes={stream_range[0]}-{stream_range[1]}"
        try:
            with self._get_http_session() as s:
                r = s.get(stream_url,
                          timeout=(self.connect_timeout, self.read_timeout),
                          headers={'Range': range_header_value})
        except Exception as e:
            self.logger.exception("can't download mds %s resource %s with exception %s" % (stream_range, stream_url, e))
            raise Exception(e)
        if r.ok:
            return r.content
        raise Exception("can't download range %s of %s" % (stream_range, stream_url))

    def calc_stream_hashes(self, stream_url: str, chunk_size=4*1024*1024):
        """
        :param stream_url: url to mds docker image source
        :param chunk_size: size of chunk
        :return: md5 sum of source, sha1 digests of chunks, summary item size
        """

        md5_pieces_hash = hashlib.md5()
        sha1_pieces_digests = []
        proccesed_pieces_size = 0
        content_size = self._get_mds_content_size(stream_url)
        ranges = SkybonedTorrentHelper.get_mds_content_ranges(content_size, chunk_size)
        for stream_range in ranges:
            piece = self._get_mds_content_with_range(stream_url, stream_range)
            md5_pieces_hash.update(piece)
            sha1_pieces_digests.append(hashlib.sha1(piece).digest())
            proccesed_pieces_size += len(piece)
        return md5_pieces_hash.hexdigest(), sha1_pieces_digests, proccesed_pieces_size

    def generate_rbtorrent(self, filename: str, mds_key: str):
        """
        :param filename: filename
        :param mds_key:  mds key
        :return: announce for added by skyboned.api resource
        """

        tvm_ticket = self._request_tvm_ticket()
        stream_url = self._get_mds_url(mds_key)
        md5_pieces_hash, sha1_pieces_digests, size = self.calc_stream_hashes(stream_url)
        items = {
            filename: {
                'type': 'file',
                'md5': md5_pieces_hash,
                'executable': False,
                'size': size
            }
        }
        hashes = {
            md5_pieces_hash: b''.join(sha1_pieces_digests)
        }
        links = {
            md5_pieces_hash: stream_url
        }
        try:
            rbt, ok = infra.skyboned.api.skyboned_add_resource(items, hashes, links, self.server, tvm_ticket)
        except Exception as e:
            raise SkybonedTorrentNotAdded(f"Failed to add rbtorrent resource {stream_url} with"
                                          f" items: {items} hashes: {hashes} links {links} and exception {e}")
        if not ok:
            raise SkybonedTorrentNotAdded(f"Failed to add rbtorrent resource {stream_url} with"
                                          f" items: {items} hashes: {hashes} links {links} with unkown exception")
        return rbt

    def remove_rbtorrent_byid(self, torrent_id: str):
        """
        :param mds_key:  mds key
        :return: result of remove resource method of skyboned.api
        """

        tvm_ticket = self._request_tvm_ticket()
        return infra.skyboned.api.skyboned_remove_resource(torrent_id, self.server, tvm_ticket)


_TORRENT_HELPERS = {
    'obsolete': ObsoleteTorrentHelper,
    'skyboned': SkybonedTorrentHelper,
}


def get_torrent_helper_from_config(conf, logger) -> IRBTorrentBrewerHelper:
    # rbtorrent helper selector

    kind = conf.get('kind', 'obsolete')
    return _TORRENT_HELPERS[kind](logger=logger, **conf.get('params', {}))
