import math


class BrewerMetrics:

    def __init__(self):
        self.data = dict()
        self._base_log = 1.5
        self._min_log = -50
        self._max_log = 50
        self._borders_log = [0.0] + [self._base_log ** x for x in range(self._min_log, self._max_log + 1)]
        self.clear()

    def _increment_metric(self, name, value: int):
        if name not in self.data:
            self.data[name] = 0
        self.data[name] += value

    def _set_metric(self, name, value: float):
        self.data[name] = value

    def _increment_manifest_metric(self, name: str, suffix: str, value: int = 1):
        name = 'torrents_manifest_{}_{}'.format(name, suffix)
        self._increment_metric(name, value)

    def _increment_brewery_metric(self, name: str, suffix: str, value: int = 1):
        name = 'torrents_brewer_{}_{}'.format(name, suffix)
        self._increment_metric(name, value)

    def _increment_event_metric(self, name: str, suffix: str, value: int = 1):
        name = 'torrents_event_{}_{}'.format(name, suffix)
        self._increment_metric(name, value)

    def append_brewery_time_histogram(self, time_seconds: float):
        name = 'torrents_brewer_time'
        self._append_log_histogram(name, time_seconds)

    def _append_log_histogram(self, name: str, value: float):
        name = '{}_{}'.format(name, 'dhhh')
        if name not in self.data:
            _buckets = dict()
            for number in self._borders_log:
                _buckets[number] = 0
            self.data[name] = _buckets
        _buckets = self.data[name]
        if value < 0:
            return
        if value == 0:
            _buckets[0.0] += 1
            return
        offset = math.floor(
            max(float(self._min_log) - 1,
                min(float(self._max_log),
                    math.log(value,
                             float(self._base_log)))) - self._min_log + 1)
        _buckets[self._borders_log[int(offset)]] += 1

    def _set_brewery_mectric(self, name: str, suffix: str, value: float = 0):
        name = 'torrents_brewer_{}_{}'.format(name, suffix)
        self._set_metric(name, value)

    # manifest
    def increment_manifest_requested_layers(self, value: int = 1):
        self._increment_manifest_metric('layers_requested', 'dmmm', value)

    def increment_manifest_torrents_layers(self, value: int = 1):
        self._increment_manifest_metric('layers_torrents', 'dmmm', value)

    def increment_manifest_skipped_layers(self, value: int = 1):
        self._increment_manifest_metric('layers_skipped', 'dmmm', value)

    def increment_manifest_queued_layers(self, value: int = 1):
        self._increment_manifest_metric('layers_queued', 'dmmm', value)

    # brewer
    def increment_brewer_migrated_layers(self, value: int = 1):
        self._increment_brewery_metric('layers_migrated', 'dmmm', value)

    def increment_brewer_queued_layers(self, value: int = 1):
        self._increment_brewery_metric('layers_queued', 'dmmm', value)

    def increment_brewer_finished_layers(self, value: int = 1):
        self._increment_brewery_metric('layers_finished', 'dmmm', value)

    def increment_brewer_started_layers(self, value: int = 1):
        self._increment_brewery_metric('layers_started', 'dmmm', value)

    def increment_brewer_dropped_layers(self, value: int = 1):
        self._increment_brewery_metric('layers_dropped', 'dmmm', value)

    def increment_brewer_errors_layers(self, value: int = 1):
        self._increment_brewery_metric('layers_errors', 'dmmm', value)

    # event
    def increment_event(self, action: str, value: int = 1):
        self._increment_event_metric('total', 'dmmm', value)
        self._increment_event_metric('{}_total'.format(action), 'dmmm', value)

    def increment_event_queued(self, value: int = 1):
        self._increment_event_metric('layers_queued', 'dmmm', value)

    def increment_event_skipped(self, value: int = 1):
        self._increment_event_metric('layers_skipped', 'dmmm', value)

    def increment_event_errors(self, value: int = 1):
        self._increment_event_metric('layers_errors', 'dmmm', value)

    def increment_event_requested(self, value: int = 1):
        self._increment_event_metric('layers_requested', 'dmmm', value)

    def clear(self):
        self.data = dict()
        self.increment_manifest_requested_layers(0)
        self.increment_manifest_skipped_layers(0)
        self.increment_manifest_torrents_layers(0)
        self.increment_manifest_queued_layers(0)
        self.increment_brewer_migrated_layers(0)
        self.increment_brewer_finished_layers(0)
        self.increment_brewer_queued_layers(0)
        self.increment_brewer_started_layers(0)
        self.increment_brewer_dropped_layers(0)
        self.increment_event('push', 0)
        self.increment_event_queued(0)
        self.increment_event_skipped(0)
        self.increment_event_errors(0)
        self.increment_event_requested(0)
        self.append_brewery_time_histogram(-1)

    def get_metrics(self):
        result = list()
        for name in sorted(self.data.keys()):
            value = self.data[name]
            if isinstance(value, dict):
                histogram = list()
                for time in sorted(value.keys()):
                    histogram.append([
                        time,
                        value[time]
                    ])
                result.append([
                    name,
                    histogram
                ])
                pass
            else:
                result.append([
                    name,
                    self.data[name]
                ])
        return result
