from infra.rtc.docker_registry.docker_torrents.exceptions import InsufficientParametersException, DublicateMapping
from infra.rtc.docker_registry.docker_torrents.clients.blackbox import BlackboxClient
from infra.rtc.docker_registry.docker_torrents.clients.registry import RegistryClient
from infra.rtc.docker_registry.docker_torrents.torrent_config import TorrentsConfig
from infra.rtc.docker_registry.docker_torrents.database import TorrentDatabase
from infra.rtc.docker_registry.docker_torrents.clients.rbtorrent_helper import IRBTorrentBrewerHelper
from infra.rtc.docker_registry.docker_torrents.metrics import BrewerMetrics
from threading import Thread
from requests import Timeout
from time import sleep, time
import random


class TorrentBrewer:

    def __init__(self,
                 config: TorrentsConfig,
                 registry: RegistryClient,
                 database: TorrentDatabase,
                 blackbox: BlackboxClient,
                 torrent_helper: IRBTorrentBrewerHelper):
        self.sleep_time = float(config.brewer.get('sleep_time', 10))
        self.discard_timeout = int(config.brewer.get('brew_discard_timeout', 600))
        self.layeres_per_brew_iteration = int(config.brewer.get('layers_per_chunk', 10))
        self.blackbox = blackbox
        self.database = database
        self.registry = registry
        self.torrent_helper = torrent_helper
        self.alive = True
        self.logger = config.logger
        self.brew_thread = BrewThread(self)
        self.brew_thread.start()
        self.metrics = BrewerMetrics()

    def process_event(self, event):
        self.metrics.increment_event(event['action'], 1)
        if event['action'] != 'push':
            return
        scope = event['target']['repository']
        digest = event['target']['digest'].split(':')[1]
        self.metrics.increment_event_requested(1)
        if not self.database.repo_has_role(scope, 'rbtorrent'):
            self.metrics.increment_event_skipped(1)
            self.logger.warning(
                'repo: {scope} has no rbtorrent role'.format(scope=scope))
            return
        try:
            self.enque_brew([digest])
            self.metrics.increment_event_queued(1)
        except Exception:
            self.metrics.increment_event_errors(1)
            self.logger.exception('Error while queuing event layer')

    def get_found_total_layers(self, manifest: dict):
        total_layers = 0
        found_layers = 0
        for layer in manifest['layers']:
            total_layers += 1
            if 'rbtorrent_id' in layer:
                found_layers += 1
        return found_layers, total_layers

    def get_manifest_with_torrents(self, scope: str, tag: str, headers: dict) -> dict:
        manifest = self.registry.resolve_manifest(scope, tag, headers)
        unfound = self.fill_manifest_with_torrents(manifest)
        found_layers, total_layers = self.get_found_total_layers(manifest)
        self.logger.info('Found {}/{} layers for {}:{}.'.format(found_layers, total_layers, scope, tag))
        if len(unfound) > 0:
            try:
                if self.request_brew_unfound(unfound, scope, headers):
                    self.fill_manifest_with_torrents(manifest)
            except Exception:
                self.logger.exception('Non fatal error during request')
        new_found_layers, total_layers = self.get_found_total_layers(manifest)
        self.metrics.increment_manifest_requested_layers(total_layers)
        self.metrics.increment_manifest_torrents_layers(new_found_layers)
        if new_found_layers != found_layers:
            self.logger.info('Returning {}/{} layers for {}:{}.'.format(found_layers, total_layers, scope, tag))
        return manifest

    def fill_manifest_with_torrents(self, manifest: dict) -> set:
        unfound = set()
        blobs = set()
        for layer in manifest['layers']:
            blobs.add(layer['digest'].split(':')[1])
        mapping = self.database.search_digest_mapping(list(blobs))
        for layer in manifest['layers']:
            digest = layer['digest'].split(':')[1]
            if digest in mapping:
                if mapping[digest] is not None:
                    layer['rbtorrent_id'] = mapping[digest]
            else:
                unfound.add(digest)
        return unfound

    def request_brew_unfound(self, unfound: set, scope: str, headers: dict):

        if len(unfound) == 0:
            return
        if self.blackbox is None:
            self.logger.warning('Ignoring rbtorrent role ckeck for scope: {}'.format(scope))
        else:
            if self.blackbox.ip_header not in headers or 'Authorization' not in headers:
                raise InsufficientParametersException(
                    '{} and Authorization headers are mandatory'.format(self.blackbox.ip_header))
            username = self.blackbox.login(headers)
            if not self.database.check_role(scope, username, 'rbtorrent'):
                self.logger.warning('{user} has no rbtorrent role on scope: {scope}'.format(user=username, scope=scope))
                self.metrics.increment_manifest_skipped_layers(len(unfound))
                return
        digest_mds_map = self.database.get_mds_keys(unfound)
        result = self.enque_brew(digest_mds_map)
        self.metrics.increment_manifest_queued_layers(len(unfound))
        return result

    def enque_brew(self, digests) -> bool:
        reload = False
        mds_map = self.database.get_mds_keys(digests)
        mds_keys = list()
        for digest in mds_map.keys():
            mds_keys.append(mds_map[digest])
        old_mds_rbtorrent_map = self.database.search_old_mds_key_mapping(mds_keys)
        for digest in digests:
            mds_key = mds_map[digest]
            try:
                if mds_key in old_mds_rbtorrent_map:
                    self.logger.info(
                        'Found layer digest: {}, mds_key: {}, rbtorrent_id: {} in old table; Putting to new'.format(
                            digest, mds_key, old_mds_rbtorrent_map[mds_key]
                        ))
                    self.metrics.increment_brewer_migrated_layers(1)
                    self.database.new_mapping(digest, old_mds_rbtorrent_map[mds_key])
                    reload = True
                else:
                    self.logger.info(
                        'Enqueueing brew digest: {}'.format(
                            digest
                        ))
                    self.database.put_queue_brew(digest)
                    self.metrics.increment_brewer_queued_layers(1)
            except DublicateMapping as error:
                self.logger.error(str(error))
                reload = True
            except Exception:
                self.logger.exception('Error while putting digest: {}, mds_key: {} to brew queue'.format(
                    digest,
                    mds_key
                ))
        return reload

    def list_brewing(self, offset: int = 0, limit: int = 1000):
        return self.database.get_brewing(offset, limit)

    def list_queued(self, offset: int = 0, limit: int = 1000):
        return self.database.get_queued(offset, limit)

    def kill(self):
        self.alive = False


class BrewThread(Thread):

    def __init__(self, brewer: TorrentBrewer):
        self.brewer = brewer
        self.logger = brewer.logger
        super().__init__()

    def run(self):
        self.logger.info('Brew thread started.')
        while self.brewer.alive:
            try:
                chunk = self.brewer.database.pop_queue(
                    self.brewer.discard_timeout,
                    self.brewer.layeres_per_brew_iteration
                )
                if len(chunk) == 0:
                    self.logger.info("No layers to brew")
                else:
                    self.logger.info("Found {} layers to brew.".format(len(chunk)))
                mds_map = self.brewer.database.get_mds_keys(chunk)
                threads = list()
                for digest in chunk:
                    mds_key = mds_map.get(digest, None)
                    thread = LayerBrewThread(self.brewer, digest, mds_key)
                    threads.append(thread)
                    thread.start()
                for thread in threads:
                    thread.join()
            except Exception:
                self.logger.exception('Exception while getting layers to brew.')
            sleep(self.brewer.sleep_time + random.randint(0, int(self.brewer.sleep_time)))
        self.logger.fatal('Brew thread ended.')


class LayerBrewThread(Thread):
    def __init__(self, brewer: TorrentBrewer, digest: str, mds_key: str):
        self.mds_key = mds_key
        self.digest = digest
        self.brewer = brewer
        self.error = None
        self.logger = brewer.logger
        super().__init__()

    def run(self):
        try:
            try:
                self.logger.info('Starting brew digest: {} mds_key: {}'.format(
                    self.digest, self.mds_key
                ))
                self.brewer.metrics.increment_brewer_started_layers(1)
                start_time = time()
                rbtorrent_id = self.brewer.torrent_helper.generate_rbtorrent(self.digest, self.mds_key)
                end_time = time()
                brew_time = end_time - start_time
                if rbtorrent_id is None:
                    raise Exception("Brew result of digest: {} mds_key: {} is null".format(
                        self.digest, self.mds_key
                    ))
                self.logger.info('Done brewing digest: {} mds_key: {} rbtorrent_id {} time spend: {} seconds'.format(
                    self.digest, self.mds_key, rbtorrent_id, brew_time
                ))
                self.brewer.database.update_mapping(self.digest, rbtorrent_id)
                self.brewer.metrics.increment_brewer_finished_layers(1)
                self.brewer.metrics.append_brewery_time_histogram(brew_time)
            except Timeout as timeout_error:
                self.logger.exception(
                    'Timeout while brewing digest: {}, mds_key: {}, will retry in ~{} seconds'.format(
                        self.digest, self.mds_key, self.brewer.sleep_time))
                self.brewer.metrics.increment_brewer_errors_layers(1)
                raise timeout_error
            except Exception as error:
                self.logger.exception(
                    'Error while brewing digest: {}, mds_key: {}, removing from queue'.format(self.digest,
                                                                                              self.mds_key))
                self.brewer.metrics.increment_brewer_errors_layers(1)
                self.brewer.database.drop_mapping(self.digest)
                self.brewer.metrics.increment_brewer_dropped_layers(1)
                raise error
        except Exception as error:
            self.error = error
