from infra.rtc.docker_registry.docker_torrents.torrent_config import TorrentsConfig
from infra.rtc.docker_registry.docker_torrents.torrent_brewer import TorrentBrewer
from infra.rtc.docker_registry.docker_torrents.exceptions import SubrequestException, BlackboxException
from flask import Flask, request, Response
import logging
import json

app = Flask(__name__)
logger = logging.getLogger(__name__)
compatibility_header = None
event_authorization = None
brewer: TorrentBrewer = None


def start(cfg: TorrentsConfig, brew: TorrentBrewer):
    global compatibility_header
    global event_authorization
    global brewer
    compatibility_header = cfg.docker.get(
        'compatibility_header',
        'application/vnd.docker.distribution.manifest.v2+json'
    )
    event_authorization = cfg.application.get('event_authorization', None)
    brewer = brew
    app.run(
        host=cfg.application.get('host', '::'),
        port=int(cfg.application.get('port', 5000)),
        debug=cfg.debug_mode
    )


@app.route('/v0/queued', methods=['GET', 'HEAD'])
def get_queued():
    try:
        data, count = brewer.list_queued()
        return Response(json.dumps(brewer.list_queued()))
    except Exception as error:
        logger.exception('Fatal error "{}" during list queued request'.format(str(error)))
        return Response(str(error), status=500)


@app.route('/v0/brewing', methods=['GET', 'HEAD'])
def get_brewing():
    try:
        data, count = brewer.list_brewing()
        return Response(json.dumps(brewer.list_brewing()))
    except Exception as error:
        logger.exception('Fatal error during list queued request')
        return Response(str(error), status=500)


@app.route('/v0/unistat', methods=['GET', 'HEAD'])
def get_unistat():
    try:
        metrics = brewer.metrics.get_metrics()
        return Response(json.dumps(metrics))
    except Exception as error:
        logger.exception('Fatal error during unistat request')
        return Response(str(error), status=500)


@app.route('/v0/event', methods=['POST', 'PUT'])
def distribution_events():
    if event_authorization is not None:
        auth = request.headers.get('Authorization', None)
        if auth != event_authorization:
            return Response('Unauthorized', status=401)
    events = request.json
    for event in events['events']:
        try:
            brewer.process_event(event)
        except Exception as error:
            logger.exception('Error "{}" during event processing'.format(str(error)))
    return Response('OK', status=200)


@app.route('/v2/<path:scope>/manifests/<tag>', methods=['GET', 'HEAD'])
def get_rbtorrents(scope, tag):
    headers = dict()
    headers.update(request.headers)
    headers.pop('Host')
    try:
        manifest = brewer.get_manifest_with_torrents(scope, tag, headers)
        resp = Response(json.dumps(manifest), headers={'Content-Type': compatibility_header})
        return resp
    except BlackboxException as blackbox_error:
        logger.exception('Fatal error during manifest request')
        return Response(str(blackbox_error), status=403)
    except SubrequestException as subrequest_error:
        logger.exception('Fatal error during manifest request')
        return Response(str(subrequest_error), status=subrequest_error.http_error.response.status_code)
    except Exception as error:
        logger.exception('Fatal error "{}" during manifest request'.format(str(error)))
        return Response(str(error), status=500)
