package fstab

import (
	"bufio"
	"fmt"
	"io"
	"os"
	"path"
	"strconv"
	"strings"
)

const DefaultPath = "/etc/fstab"

var supportedDevices = map[string]string{
	"UUID":      "by-uuid",
	"PARTUUID":  "by-partuuid",
	"PATH":      "by-path",
	"LABEL":     "by-label",
	"PARTLABEL": "by-partlabel",
	"ID":        "by-id",
}

type Entry struct {
	DevType string
	DevName string
	FSName  string
	Dir     string
	Type    string
	Opts    string
	Freq    int
	Pass    int
}

func ParseFromPath(fstabPath string) ([]*Entry, error) {
	f, err := os.Open(fstabPath)
	if err != nil {
		return nil, err
	}
	defer f.Close()
	return ParseFromReader(f)
}

func ParseFromReader(r io.Reader) ([]*Entry, error) {
	var result []*Entry
	scanner := bufio.NewScanner(r)
	for scanner.Scan() {
		if err := scanner.Err(); err != nil {
			return nil, err
		}
		line := scanner.Text()
		// skip comments or too short lines
		if strings.HasPrefix(line, "#") || len(line) < 12 {
			continue
		}
		fields := strings.Fields(line)
		if len(fields) != 6 {
			return nil, fmt.Errorf("failed to parse line '%s', unexpected field number %d, expecting exactly 6 fields", line, len(fields))
		}

		var devType, devName string
		// assume that we do not have any strange devices UUID=ZFS=smth
		devFields := strings.Split(fields[0], "=")
		if len(devFields) == 2 {
			if p, ok := supportedDevices[devFields[0]]; ok {
				devType = devFields[0]
				devName = path.Join("/dev/disk", p, devFields[1])
			} else {
				return nil, fmt.Errorf("unsuuported device type %s", devFields[0])
			}
		} else {
			devType = "file"
			devName = fields[0]
		}

		freq, err := strconv.ParseInt(fields[4], 10, 32)
		if err != nil {
			return nil, err
		}
		pass, err := strconv.ParseInt(fields[5], 10, 32)
		if err != nil {
			return nil, err
		}

		result = append(result, &Entry{
			DevType: devType,
			DevName: devName,
			FSName:  fields[0],
			Dir:     fields[1],
			Type:    fields[2],
			Opts:    fields[3],
			Freq:    int(freq),
			Pass:    int(pass),
		})
	}
	return result, nil
}

func (e *Entry) String() string {
	return fmt.Sprintf("FSTAB Entry(dev file %s): %s %s %s %s %d %d", e.DevName, e.FSName, e.Dir, e.Type, e.Opts, e.Freq, e.Pass)
}
