package fstab

import (
	"strings"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestParseFromReader_Ok(t *testing.T) {
	reader := strings.NewReader(`
proc	/proc		proc	nodev,noexec,nosuid	0	0
# / was on /dev/sdc2 during installation
UUID=18b666da-1bc0-43c7-a453-a0ede0e42478	/		ext4	barrier=1,noatime,lazytime	0	1
# /home was on /dev/sdc3 during installation
UUID=d86184d3-14ef-45cd-985b-48633a45f33e	/home		ext4	barrier=1,noatime,lazytime,nosuid,nodev	0	2
# /place was on /dev/sdc4 during installation
UUID=278629d7-07c2-47e6-a2f0-f01da2538a79	/place		ext4	barrier=1,noatime,lazytime,nosuid,nodev	0	2
# /ssd was on /dev/md5 during installation
UUID=1a165d19-e367-4c01-bac1-7973dac3da4f	/ssd		ext4	barrier=1,noatime,lazytime,nosuid,nodev	0	2
`)
	expected := []*Entry{
		{
			DevType: "file",
			DevName: "proc",
			FSName:  "proc",
			Dir:     "/proc",
			Type:    "proc",
			Opts:    "nodev,noexec,nosuid",
			Freq:    0,
			Pass:    0,
		},
		{
			DevType: "UUID",
			DevName: "/dev/disk/by-uuid/18b666da-1bc0-43c7-a453-a0ede0e42478",
			FSName:  "UUID=18b666da-1bc0-43c7-a453-a0ede0e42478",
			Dir:     "/",
			Type:    "ext4",
			Opts:    "barrier=1,noatime,lazytime",
			Freq:    0,
			Pass:    1,
		},
		{
			DevType: "UUID",
			DevName: "/dev/disk/by-uuid/d86184d3-14ef-45cd-985b-48633a45f33e",
			FSName:  "UUID=d86184d3-14ef-45cd-985b-48633a45f33e",
			Dir:     "/home",
			Type:    "ext4",
			Opts:    "barrier=1,noatime,lazytime,nosuid,nodev",
			Freq:    0,
			Pass:    2,
		},
		{
			DevType: "UUID",
			DevName: "/dev/disk/by-uuid/278629d7-07c2-47e6-a2f0-f01da2538a79",
			FSName:  "UUID=278629d7-07c2-47e6-a2f0-f01da2538a79",
			Dir:     "/place",
			Type:    "ext4",
			Opts:    "barrier=1,noatime,lazytime,nosuid,nodev",
			Freq:    0,
			Pass:    2,
		},
		{
			DevType: "UUID",
			DevName: "/dev/disk/by-uuid/1a165d19-e367-4c01-bac1-7973dac3da4f",
			FSName:  "UUID=1a165d19-e367-4c01-bac1-7973dac3da4f",
			Dir:     "/ssd",
			Type:    "ext4",
			Opts:    "barrier=1,noatime,lazytime,nosuid,nodev",
			Freq:    0,
			Pass:    2,
		},
	}
	r, err := ParseFromReader(reader)
	assert.NoError(t, err)
	assert.Equal(t, expected, r)
}

func TestParseFromReader_Empty(t *testing.T) {
	reader := strings.NewReader("")
	r, err := ParseFromReader(reader)
	assert.NoError(t, err)
	assert.Nil(t, r)

	reader = strings.NewReader("\n\n\n\n\n")
	r, err = ParseFromReader(reader)
	assert.NoError(t, err)
	assert.Nil(t, r)
}

func TestParseFromReader_Malformed_Line(t *testing.T) {
	reader := strings.NewReader("this is malformed line")
	r, err := ParseFromReader(reader)
	assert.Error(t, err)
	assert.Nil(t, r)
}

func TestParseFromReader_Malformed_Freq(t *testing.T) {
	reader := strings.NewReader("proc\t/proc\t\tproc\tnodev,noexec,nosuid\tmess\t0")
	r, err := ParseFromReader(reader)
	assert.Error(t, err)
	assert.Nil(t, r)
}

func TestParseFromReader_Malformed_Pass(t *testing.T) {
	reader := strings.NewReader("proc\t/proc\t\tproc\tnodev,noexec,nosuid\t0\tmess")
	r, err := ParseFromReader(reader)
	assert.Error(t, err)
	assert.Nil(t, r)
}

func TestParseFromReader_Unsupported_DevType(t *testing.T) {
	reader := strings.NewReader("ZFS=proc\t/proc\t\tproc\tnodev,noexec,nosuid\t0\t0")
	r, err := ParseFromReader(reader)
	assert.Error(t, err)
	assert.Nil(t, r)
}
