#!/usr/bin/env python
# -*- coding: utf-8 -*-

import argparse
import logging
import json
from collections import namedtuple

from utils import init_root_logger, run_updater, MonitoringUpdater, read_objects_from_folder, read_objects_from_files
from utils.exceptions import MonitoringUpdaterHttpError

log = logging.getLogger(__name__)

ALERTS_REPLACE_API = "srvambry/alerts/replace"

ALERT_TEMPLATE_GET_API = 'srvambry/tmpl/alerts/get?key={}'.format
ALERT_TEMPLATE_CREATE_API = 'srvambry/tmpl/alerts/create?key={}'.format
ALERT_TEMPLATE_UPDATE_API = 'srvambry/tmpl/alerts/update?key={}'.format
ALERT_TEMPLATE_APPLY_API = 'srvambry/tmpl/alerts/apply/{}'.format
ALERT_TEMPLATE_RENDER_API = 'srvambry/tmpl/alerts/render_json/{}'.format

MANUAL_ALERTS_FOLDER = 'alerts/manual'
TEMPLATE_ALERTS_FOLDER = 'alerts/templated'

TARGETS = ['prod']

MANUAL_ALERTS_PREFIX = 'rtc_manual'
TEMPLATE_ALERTS_PREFIX = 'rtc_templated'

OWNERS = ['dldmitry', 'qwizzy', 'ndolganov']


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-a', '--template-alerts', type=str, nargs='+')
    parser.add_argument('--dry-run', action='store_true', default=False)

    return parser.parse_args()


ManualAlert = namedtuple('ManualAlert', ('key', 'data', 'path'))


class TemplateAlert(object):
    def __init__(self, key, content, path, is_periodic=True, owners=None):
        if owners is None:
            owners = OWNERS

        self.key = '{}.{}'.format(TEMPLATE_ALERTS_PREFIX, key)
        self.content = content
        self.owners = owners
        self.should_apply = not key.startswith("_")
        self.is_periodic = self.should_apply and is_periodic
        self.path = path

    def to_dict(self):
        return {
            'key': self.key,
            'owners': self.owners,
            'content': self.content,
            'is_periodic': self.is_periodic,
            'abc': 'srertc'
        }


class AlertsUpdater(MonitoringUpdater):
    def __init__(self, template_alerts=None, dry_run=False):
        self._manual_alerts = read_objects_from_folder(ManualAlert, MANUAL_ALERTS_FOLDER, json.loads)

        if template_alerts is None:
            self._template_alerts = read_objects_from_folder(TemplateAlert, TEMPLATE_ALERTS_FOLDER)
        else:
            self._template_alerts = read_objects_from_files(TemplateAlert, template_alerts)

        self._template_alerts = sorted(self._template_alerts, key=lambda alert: alert.key)

        self._dry_run = dry_run

    @classmethod
    def _replace_alerts_by_prefix(cls, target, alerts, alerts_names):
        try:
            cls._post(cls._compose_url(target, ALERTS_REPLACE_API), json=alerts)
            log.info(
                "Replaced '%s' manual alerts by prefix '%s' on '%s' contour",
                alerts_names,
                MANUAL_ALERTS_PREFIX,
                target
            )
        except MonitoringUpdaterHttpError as e:
            log.error(
                "Error when replacing '%s' manual alerts using %s\n%s",
                alerts_names, e.url, e.traceback or e.error
            )

    @classmethod
    def _check_template_alert_existence(cls, target, alert_key):
        return cls._check_object_existence(target, ALERT_TEMPLATE_GET_API(alert_key))

    @classmethod
    def _create_template_alert(cls, target, alert):
        try:
            cls._post(cls._compose_url(target, ALERT_TEMPLATE_CREATE_API(alert.key)), json=alert.to_dict())
            log.info("Created '%s' template alert on '%s' contour", alert.key, target)
        except MonitoringUpdaterHttpError as e:
            log.error(
                "Error when creating alert '%s' from file '%s' using %s\n%s",
                alert.key, alert.path, e.url, e.traceback or e.error
            )

    @classmethod
    def _update_template_alert(cls, target, alert):
        try:
            cls._post(cls._compose_url(target, ALERT_TEMPLATE_UPDATE_API(alert.key)), json=alert.to_dict())
            log.debug("Updated '%s' template alert on '%s' contour", alert.key, target)
        except MonitoringUpdaterHttpError as e:
            log.error(
                "Error when updating alert '%s' from file '%s' using %s\n%s",
                alert.key, alert.path, e.url, e.traceback or e.error
            )

    @classmethod
    def _apply_template_alert(cls, target, alert):
        try:
            cls._post(cls._compose_url(target, ALERT_TEMPLATE_APPLY_API(alert.key)), json_format=False)
            log.info("Applied '%s' template alert on '%s' contour", alert.key, target)
        except MonitoringUpdaterHttpError as e:
            log.error(
                "Error when applying alert '%s' from file '%s' using %s\n%s",
                alert.key, alert.path, e.url, e.traceback or e.error
            )

    @classmethod
    def _render_template_alert(cls, target, alert):
        try:
            rendered_template = cls._get(cls._compose_url(target, ALERT_TEMPLATE_RENDER_API(alert.key)))
            log.info(
                "Rendered '%s' template alert on '%s' contour\n%s",
                alert.key, target, json.dumps(rendered_template, indent=4)
            )
        except MonitoringUpdaterHttpError as e:
            log.error(
                "Error when rendering alert '%s' from file '%s' using %s\n%s",
                alert.key, alert.path, e.url, e.traceback or e.error
            )

    def _upload_manual_alerts(self, target):
        if self._dry_run:
            return

        alerts = {
            "prefix": MANUAL_ALERTS_PREFIX,
            "alerts": [x.data for x in self._manual_alerts],
        }
        alerts_names = ', '.join([x.data['name'] for x in self._manual_alerts])

        if alerts_names:
            self._replace_alerts_by_prefix(target, alerts, alerts_names)

    def _upload_template_alerts(self, target):
        for alert in self._template_alerts:
            if self._check_template_alert_existence(target, alert.key):
                self._update_template_alert(target, alert)
            else:
                self._create_template_alert(target, alert)

            if alert.should_apply:
                if self._dry_run:
                    self._render_template_alert(target, alert)
                else:
                    self._apply_template_alert(target, alert)

    def run(self):
        for target in TARGETS:
            self._upload_manual_alerts(target)
            self._upload_template_alerts(target)


if __name__ == '__main__':
    init_root_logger()
    args = parse_args()

    run_updater(AlertsUpdater(args.template_alerts, args.dry_run), log)
