#!/usr/bin/env python
# -*- coding: utf-8 -*-

import argparse
import logging
from collections import namedtuple

from utils.exceptions import MonitoringUpdaterHttpError
from utils import init_root_logger, run_updater, MonitoringUpdater, read_objects_from_folder, read_objects_from_files

log = logging.getLogger(__name__)

PANEL_TEMPLATE_GET_API = 'srvambry/tmpl/panels/get?key={}'.format
PANEL_TEMPLATE_CREATE_API = 'srvambry/tmpl/panels/create'
PANEL_TEMPLATE_UPDATE_API = 'srvambry/tmpl/panels/update/content?key={}'.format

PANELS_FOLDER = 'panels'

TARGETS = ['prod']

OWNERS = ['dldmitry']

Panel = namedtuple('Panel', ['key', 'data', 'path'])


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-p', '--template-panels', type=str, nargs='+')

    return parser.parse_args()


class PanelsUpdater(MonitoringUpdater):
    def __init__(self, template_panels=None, dry_run=None):
        if template_panels is None:
            self._template_panels = read_objects_from_folder(Panel, PANELS_FOLDER)
        else:
            self._template_panels = read_objects_from_files(Panel, template_panels)

    @classmethod
    def _check_template_panel_existence(cls, target, panel_key):
        return cls._check_object_existence(target, PANEL_TEMPLATE_GET_API(panel_key))

    @classmethod
    def _create_template_panel(cls, target, new_panel):
        try:
            cls._post(cls._compose_url(target, PANEL_TEMPLATE_CREATE_API), json=new_panel)
            log.info("Created '%s' template panel on '%s' contour", new_panel['key'], target)
        except MonitoringUpdaterHttpError as e:
            log.error(
                "Error when creating new template panel '%s' on %s\n%s",
                new_panel['key'], e.url, e.traceback or e.error
            )

    @classmethod
    def _udpate_template_panel(cls, target, panel):
        try:
            cls._post(cls._compose_url(target, PANEL_TEMPLATE_UPDATE_API(panel.key)), data=panel.data, json_format=False)
            log.info("Updated '%s' template panel on '%s' contour", panel.key, target)
        except MonitoringUpdaterHttpError as e:
            log.error(
                "Error when updating template panel '%s' from file '%s' using %s\n%s",
                panel.key, panel.path, e.url, e.traceback or e.error
            )

    def _upload_template_panels(self, target):
        for panel in self._template_panels:
            if not self._check_template_panel_existence(target, panel.key):
                new_panel = {
                    "owners": OWNERS,
                    "key": panel.key,
                    "abc": "srertc"
                }
                self._create_template_panel(target, new_panel)

            self._udpate_template_panel(target, panel)

    def run(self):
        for target in TARGETS:
            self._upload_template_panels(target)


if __name__ == '__main__':
    init_root_logger()
    args = parse_args()

    run_updater(PanelsUpdater(args.template_panels), log)
