#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os
from collections import namedtuple
import json
import logging

from utils import init_root_logger, run_updater, MonitoringUpdater, check_path_existence
from utils.exceptions import MonitoringUpdaterHttpError

log = logging.getLogger(__name__)

TREE_REPLACE_API = 'srvambry/menutree/replace'

TREES_FOLDER = 'trees'

Tree = namedtuple('Tree', ['key', 'data', 'path'])


class TreesUpdater(MonitoringUpdater):
    def __init__(self, dry_run=None):
        self._trees = self._get_trees(TREES_FOLDER)

    @classmethod
    def _get_trees(cls, trees_folder):
        trees = []

        check_path_existence(trees_folder)

        for tree_path, dirs, targets_paths in os.walk(trees_folder):
            for target_path in targets_paths:
                file_path = os.path.join(tree_path, target_path)
                with open(file_path) as tree_file:
                    tree_data = json.load(tree_file)

                trees.append(Tree(target_path, tree_data, file_path))

        return trees

    @classmethod
    def _replace_tree(cls, tree, target="prod"):
        try:
            cls._post(cls._compose_url(target, TREE_REPLACE_API), json=tree.data)
            log.info("Replaced '%s' tree on '%s' contour", tree.key, target)
        except MonitoringUpdaterHttpError as e:
            log.error(
                "Error when replacing tree '%s' from file '%s' using %s\n%s",
                tree.key, tree.path, e.url, e.traceback or e.error
            )

    def run(self):
        for tree in self._trees:
            self._replace_tree(tree)


if __name__ == '__main__':
    init_root_logger()

    run_updater(TreesUpdater(), log)
