# -*- coding: utf-8 -*-

import sys
import logging
import os

from .exceptions import MonitoringUpdaterNonExistentPath, MonitoringUpdaterBaseError


def init_root_logger():
    logging.basicConfig(
        stream=sys.stderr,
        level=logging.INFO,
        format='%(asctime)s %(levelname)s (%(module)s) %(message)s',
    )


def read_objects_from_folder(klass, folder_path, load_func=None):
    file_paths = [
        os.path.join(folder_path, file_path)
        for file_path in os.listdir(folder_path)
        if not file_path.endswith(".swp")
    ]

    return read_objects_from_files(klass, file_paths, load_func)


def read_objects_from_files(klass, file_paths, load_func=None):
    objects = []

    for file_path in file_paths:
        check_path_existence(file_path)

        with open(file_path) as object_file:
            object_data = object_file.read()

            if load_func:
                object_data = load_func(object_data)

        objects.append(klass(os.path.basename(file_path), object_data, file_path))

    return objects


def check_path_existence(path):
    if not os.path.exists(path):
        raise MonitoringUpdaterNonExistentPath('Path "{}" does not exists'.format(path))


def run_updater(updater, log):
    try:
        updater.run()
    except MonitoringUpdaterBaseError as e:
        log.error(e)
        sys.exit(1)
    except Exception as e:
        log.exception(e)
        sys.exit(1)
