# -*- coding: utf-8 -*-

from abc import ABCMeta, abstractmethod
from urlparse import urljoin

import requests

from .exceptions import MonitoringUpdaterHttpError

YASM_HOST_PROD = 'http://yasm.yandex-team.ru'


class MonitoringUpdater(object):
    __metaclass__ = ABCMeta

    HEADERS = {
        'Content-Type': 'application/json',
    }

    YASM_HOST = {
        'prod': YASM_HOST_PROD
    }

    @abstractmethod
    def run(self):
        raise NotImplementedError

    @classmethod
    def _get(cls, url, json_format=True):
        response = requests.get(url, headers=cls.HEADERS)

        try:
            response.raise_for_status()
        except requests.HTTPError as e:
            raise cls._process_error(e, url)

        if json_format:
            return response.json()

        return response

    @classmethod
    def _post(cls, url, json=None,  data=None, json_format=True):
        if data is None:
            data = {}

        response = requests.post(url, data=data, json=json, headers=cls.HEADERS)
        try:
            response.raise_for_status()
        except requests.HTTPError as e:
            raise cls._process_error(e, url)

        if json_format:
            return response.json()

        return response

    @classmethod
    def _process_error(cls, error, url):
        try:
            response = error.response.json()
            error_text = response['error']
            traceback = response.get('traceback', '').rstrip()
        except Exception:
            error_text = str(error)
            traceback = ''

        return MonitoringUpdaterHttpError(
            'HTTP request to {} failed'.format(url),
            error=error_text,
            traceback=traceback,
            url=url,
            status_code=error.response.status_code,
        )

    @classmethod
    def _compose_url(cls, target, api):
        return urljoin(cls.YASM_HOST[target], api)

    @classmethod
    def _check_object_existence(cls, target, object_path):
        try:
            cls._get(cls._compose_url(target, object_path))
        except MonitoringUpdaterHttpError as e:
            if e.status_code == 404:
                return False

            raise
        return True
